<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Sniffs\Header;

use PHP_CodeSniffer\Files\File;
use PHP_CodeSniffer\Sniffs\Sniff;

class CopyrightHeaderSniff implements Sniff
{
    public string $copyrightContent = <<<'COPYRIGHT'
/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */
COPYRIGHT;

    public function register()
    {
        return [T_OPEN_TAG];
    }

    public function process(File $phpcsFile, $stackPtr)
    {
        if ($stackPtr !== 0) {
            return;
        }

        $tokens = $phpcsFile->getTokens();

        if (($copyrightPtr = $phpcsFile->findNext([T_WHITESPACE], $stackPtr + 1, null, true))
            && $tokens[$copyrightPtr]['code'] === T_DOC_COMMENT_OPEN_TAG
        ) {
            $copyrightPtrStart = $copyrightPtr;
            $comment = $tokens[$copyrightPtr]['content'];
            while ($tokens[$copyrightPtr]['code'] !== T_DOC_COMMENT_CLOSE_TAG) {
                $comment .= $tokens[++$copyrightPtr]['content'];
            }

            if ($comment !== $this->copyrightContent) {
                $fix = $phpcsFile->addFixableError('Copyright comment content does not match', $stackPtr, 'CopyrightCommentNotMatch');

                if ($fix === true) {
                    foreach (range($copyrightPtrStart, $copyrightPtr - 1) as $ptr) {
                        $phpcsFile->fixer->replaceToken($ptr, '');
                    }
                    $phpcsFile->fixer->replaceToken($copyrightPtr, $this->copyrightContent);
                }
            }
        } else {
            $fix = $phpcsFile->addFixableError('No copyright comment found', $stackPtr, 'NoCopyrightComment');

            if ($fix === true) {
                $phpcsFile->fixer->addContent($stackPtr, "\n" . $this->copyrightContent . "\n");
            }
        }
    }
}
