<?php

namespace XCart\Extender\Tests\Action;

use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use XCart\Extender\Action\ReflectorCached;
use XCart\Extender\Action\Reflector;
use XCart\Extender\Mapping\Extender\After;
use XCart\Extender\Mapping\Extender\Depend;
use XCart\Extender\Model\Entity;
use XCart\Extender\Factory\ReflectionFactoryInterface;

/**
 * @covers \XCart\Extender\Action\ReflectorCached
 */
class ReflectorCachedTest extends TestCase
{
    public function testIsClassSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('isClass')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->isClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertTrue($cachedReflector->isClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testIsClassMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('isClass')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->isClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertTrue($cachedReflector->isClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertTrue($cachedReflector->isClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetParentSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('getParent')
            ->withAnyParameters()->willReturn('ParentClass');

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals('ParentClass', $cachedReflector->getParent(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals('ParentClass', $cachedReflector->getParent(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetParentMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('getParent')
            ->withAnyParameters()->willReturn('ParentClass');

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals('ParentClass', $cachedReflector->getParent(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals('ParentClass', $cachedReflector->getParent(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertEquals('ParentClass', $cachedReflector->getParent(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testIsMetaDecoratorSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('isMixin')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->isMixin(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertTrue($cachedReflector->isMixin(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testIsMetaDecoratorMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('isMixin')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->isMixin(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertTrue($cachedReflector->isMixin(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertTrue($cachedReflector->isMixin(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testHasAnnotationOfClassSingleEntitySingleClass(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('hasAnnotationOfClass')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), Depend::class));
        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), Depend::class));
    }

    public function testHasAnnotationOfClassSingleEntityMultipleClasses(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('hasAnnotationOfClass')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), Depend::class));
        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), After::class));

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), Depend::class));
    }

    public function testHasAnnotationOfClassMultipleEntitiesMultipleClasses(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(4))
            ->method('hasAnnotationOfClass')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), Depend::class));
        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), After::class));

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1'), Depend::class));

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2'), Depend::class));
        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2'), After::class));

        self::assertTrue($cachedReflector->hasAnnotationOfClass(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2'), Depend::class));
    }

    public function testGetDependenciesSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('getDependencies')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getDependencies(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getDependencies(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetDependenciesMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('getDependencies')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getDependencies(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getDependencies(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertEquals([], $cachedReflector->getDependencies(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetIncompatiblesSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('getIncompatibles')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getIncompatibles(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getIncompatibles(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetIncompatiblesMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('getIncompatibles')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getIncompatibles(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getIncompatibles(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertEquals([], $cachedReflector->getIncompatibles(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetBeforeModulesSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('getBeforeModules')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getBeforeModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getBeforeModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetBeforeModulesMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('getBeforeModules')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getBeforeModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getBeforeModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertEquals([], $cachedReflector->getBeforeModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetAfterModulesSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('getAfterModules')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getAfterModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getAfterModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetAfterModulesMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('getAfterModules')
            ->withAnyParameters()->willReturn([]);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals([], $cachedReflector->getAfterModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals([], $cachedReflector->getAfterModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertEquals([], $cachedReflector->getAfterModules(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testIsCompatibleSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('isCompatible')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->isCompatible(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertTrue($cachedReflector->isCompatible(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testIsCompatibleMultipleEntities(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('isCompatible')
            ->withAnyParameters()->willReturn(true);

        $cachedReflector = new ReflectorCached($reflector);

        self::assertTrue($cachedReflector->isCompatible(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertTrue($cachedReflector->isCompatible(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertTrue($cachedReflector->isCompatible(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetModuleSingleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::once())
            ->method('getModule')
            ->withAnyParameters()->willReturn('');

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals('', $cachedReflector->getModule(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals('', $cachedReflector->getModule(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }

    public function testGetModuleMultipleEntity(): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflector = $this->createMock(Reflector::class);
        $reflector->expects(self::exactly(2))
            ->method('getModule')
            ->withAnyParameters()->willReturn('');

        $cachedReflector = new ReflectorCached($reflector);

        self::assertEquals('', $cachedReflector->getModule(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
        self::assertEquals('', $cachedReflector->getModule(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class2')));

        self::assertEquals('', $cachedReflector->getModule(new Entity('XCart\MetaDecorator\Tests\Action\ParserData\Class1')));
    }
}
