<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Factory;

use XCart\Extender\CodeGenerator\NameMutation;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetMap;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Model\Entity;

use function array_pop;
use function explode;
use function implode;

class EntityFactory implements EntityFactoryInterface
{
    /**
     * @var SourceMapInterface
     */
    private SourceMapInterface $sourceMap;

    /**
     * @var TargetMap
     */
    private TargetMap $targetMap;

    /**
     * @param SourceMapInterface $sourceMap
     * @param TargetMap          $targetMap
     */
    public function __construct(SourceMapInterface $sourceMap, TargetMap $targetMap)
    {
        $this->sourceMap = $sourceMap;
        $this->targetMap = $targetMap;
    }

    /**
     * @param string $fqn
     * @param string $sourcePath
     *
     * @return Entity
     * @throws LogicException
     */
    public function build(string $fqn, string $sourcePath = ''): Entity
    {
        $sourcePath = $sourcePath ?: $this->sourceMap->suggestPathByFqn($fqn);
        $targetPath = $this->targetMap->suggestPathByFqn($fqn);

        return new Entity($fqn, $sourcePath, $targetPath);
    }

    /**
     * @param Entity $entity
     *
     * @return Entity
     */
    public function buildAncestor(Entity $entity): Entity
    {
        $entity = clone $entity;

        $nameParts   = explode('\\', $entity->getFqn());
        $name        = array_pop($nameParts) . 'Ancestor';
        $nameParts[] = $name;

        $mutation = new NameMutation($name);
        $entity->addMutation($mutation);

        $fqn = implode('\\', $nameParts);
        $entity->setFqn($fqn);

        $targetPath = $this->targetMap->suggestPathByFqn($fqn);
        $entity->setTargetPath($targetPath);
        $entity->setTargetStream($targetPath);

        return $entity;
    }

    /**
     * @param Entity $entity
     *
     * @return Entity
     */
    public function buildDescendant(Entity $entity): Entity
    {
        return clone $entity;
    }
}
