<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Event;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use XCart\Extender\Action\MixinLookupCached;
use XCart\Extender\Action\MixinLookupInterface;
use XCart\Extender\Autoloader\Development\StreamWrapper;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\EntityFactoryInterface;
use XCart\Extender\Model\EntityEvent;

use function count;

class DevelopmentModeSubscriber implements EventSubscriberInterface
{
    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var MixinLookupInterface
     */
    private MixinLookupInterface $mixinLookup;

    /**
     * @var StreamWrapper
     */
    private StreamWrapper $commonStreamWrapper;

    /**
     * @var StreamWrapper
     */
    private StreamWrapper $ancestorStreamWrapper;

    /**
     * @param EntityFactoryInterface $entityFactory
     * @param MixinLookupInterface   $mixinLookup
     * @param StreamWrapper          $commonStreamWrapper
     * @param StreamWrapper          $ancestorStreamWrapper
     */
    public function __construct(
        EntityFactoryInterface $entityFactory,
        MixinLookupInterface $mixinLookup,
        StreamWrapper $commonStreamWrapper,
        StreamWrapper $ancestorStreamWrapper
    ) {
        $this->entityFactory         = $entityFactory;
        $this->mixinLookup           = $mixinLookup;
        $this->commonStreamWrapper   = $commonStreamWrapper;
        $this->ancestorStreamWrapper = $ancestorStreamWrapper;
    }

    /**
     * @return array
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'build-chain'                   => ['onBuildChain', 100],
            'build-unaltered'               => 'onBuildUnaltered',
            'build-ancestor-from-source'    => 'onBuildAncestorFromSource',
            'build-mixin-from-source'       => 'onBuildMixinFromSource',
            'build-descendant-from-scratch' => 'onBuildDescendantFromScratch',
        ];
    }

    /**
     * @param EntityEvent $event
     *
     * @throws LogicException
     */
    public function onBuildChain(EntityEvent $event): void
    {
        $entity = $event->getEntity();

        if (
            $this->mixinLookup instanceof MixinLookupCached
            && ($mixins = $this->mixinLookup->getSavedMixins($entity)) !== null
        ) {
            $result = [];

            if (count($mixins) > 0) {
                $ancestor = $this->entityFactory->buildAncestor($entity);

                $result[$ancestor->getFqn()] = $this->ancestorStreamWrapper->wrap($ancestor->getSourcePath());
            }

            foreach ($mixins as $mixin) {
                $result[$mixin->getFqn()] = $this->commonStreamWrapper->wrap($mixin->getSourcePath());
            }

            $result[$entity->getFqn()] = $this->commonStreamWrapper->wrap($entity->getSourcePath());

            $event->setResult($result);
            $event->stopPropagation();
        }
    }

    /**
     * @param EntityEvent $event
     */
    public function onBuildUnaltered(EntityEvent $event): void
    {
        $entity = $event->getEntity();

        $entity->setTargetStream($this->commonStreamWrapper->wrap($entity->getSourcePath()));
    }

    /**
     * @param EntityEvent $event
     */
    public function onBuildAncestorFromSource(EntityEvent $event): void
    {
        $entity = $event->getEntity();

        $entity->setTargetStream($this->ancestorStreamWrapper->wrap($entity->getSourcePath()));
    }

    /**
     * @param EntityEvent $event
     */
    public function onBuildMixinFromSource(EntityEvent $event): void
    {
        $entity = $event->getEntity();

        $entity->setTargetStream($this->commonStreamWrapper->wrap($entity->getSourcePath()));
    }

    /**
     * @param EntityEvent $event
     */
    public function onBuildDescendantFromScratch(EntityEvent $event): void
    {
        $entity = $event->getEntity();

        $entity->setTargetStream($this->commonStreamWrapper->wrap($entity->getSourcePath()));
    }
}
