<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Autoloader\Development;

use function get_class;
use function stream_filter_register;
use function strlen;
use function substr;

class StreamWrapper
{
    /**
     * @var string
     */
    private string $filterId;

    /**
     * @var int|null
     */
    private ?int $prefixLength = null;

    /**
     * @param string                $filterId
     * @param StreamFilterInterface $filter
     */
    public function __construct(
        string $filterId,
        StreamFilterInterface $filter
    ) {
        $this->filterId = $filterId;

        if (stream_filter_register($filterId, get_class($filter))) {
            $filter::setWrapper($this);
        }
    }

    /**
     * @param string $path
     *
     * @return string
     */
    public function wrap(string $path): string
    {
        return $this->getPrefix() . $path;
    }

    /**
     * @param string $path
     *
     * @return string
     */
    public function unwrap(string $path): string
    {
        return substr($path, $this->getPrefixLength());
    }

    /**
     * @return string
     */
    private function getPrefix(): string
    {
        return "php://filter/read={$this->filterId}/resource=";
    }

    /**
     * @return int
     */
    private function getPrefixLength(): int
    {
        if ($this->prefixLength === null) {
            $this->prefixLength = strlen($this->getPrefix());
        }

        return $this->prefixLength;
    }
}
