<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Autoloader\Development;

use php_user_filter;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetMap;

use function feof;
use function file_get_contents;
use function stream_bucket_append;
use function stream_bucket_make_writeable;
use function stream_bucket_new;
use function stream_get_meta_data;

/**
 * @property $stream
 */
class AncestorStreamFilter extends php_user_filter implements StreamFilterInterface
{
    /**
     * @var StreamWrapper[]
     */
    protected static array $streamWrapper;

    /**
     * @var SourceMapInterface
     */
    protected static SourceMapInterface $sourceMap;

    /**
     * @var TargetMap
     */
    protected static TargetMap $targetMap;

    /**
     * @param StreamWrapper $streamWrapper
     */
    public static function setWrapper(StreamWrapper $streamWrapper): void
    {
        static::$streamWrapper[static::class] = $streamWrapper;
    }

    /**
     * @param SourceMapInterface $sourceMap
     */
    public static function setSourceMap(SourceMapInterface $sourceMap): void
    {
        static::$sourceMap = $sourceMap;
    }

    /**
     * @param TargetMap $targetMap
     */
    public static function setTargetMap(TargetMap $targetMap): void
    {
        static::$targetMap = $targetMap;
    }

    /**
     * @param resource $in
     * @param resource $out
     * @param int      $consumed
     * @param bool     $closing
     *
     * @return int
     */
    public function filter($in, $out, &$consumed, $closing): int
    {
        /** @noinspection PhpStatementHasEmptyBodyInspection */
        /** @noinspection LoopWhichDoesNotLoopInspection */
        /** @noinspection MissingOrEmptyGroupStatementInspection */
        while ($bucket = stream_bucket_make_writeable($in)) {
        }

        if ($closing || feof($this->stream)) {
            $streamUri = stream_get_meta_data($this->stream)['uri'];

            $source = static::$streamWrapper[static::class]->unwrap($streamUri);

            $buffer = file_get_contents($this->getTargetPath($source));

            $bucket = stream_bucket_new($this->stream, $buffer);
            stream_bucket_append($out, $bucket);

            return PSFS_PASS_ON;
        }

        return PSFS_FEED_ME;
    }

    /**
     * @param string $sourcePath
     *
     * @return string
     */
    protected function getTargetPath(string $sourcePath): string
    {
        $fqn = static::$sourceMap->suggestFqnByPath($sourcePath);

        return static::$targetMap->suggestPathByFqn($fqn . 'Ancestor');
    }
}
