<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action\ViewListReader;

use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use XCart\Extender\Action\ReflectorInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Model\Entity;
use XCart\Extender\Model\EntityEvent;

class ViewListExtractor implements ViewListExtractorInterface
{
    /**
     * @var ReflectorInterface
     */
    private ReflectorInterface $reflector;

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $eventDispatcher;

    /**
     * @param ReflectorInterface       $reflector
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        ReflectorInterface $reflector,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->reflector       = $reflector;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function read(Entity $entity): array
    {
        $viewListEvent = new EntityEvent($entity);
        $this->eventDispatcher->dispatch($viewListEvent, 'read-view-list');

        if (($result = $viewListEvent->getResult()) !== null) {
            return $result;
        }

        $result = [];
        foreach ($this->reflector->getViewListAnnotations($entity) as $annotation) {
            $record = [
                'child'  => $entity->getFqn(),
                'list'   => $annotation->list,
                'weight' => $annotation->weight,
                'name'   => $annotation->name,
            ];

            if ($annotation->zone) {
                $record['zone'] = $annotation->zone;
            }

            if ($annotation->preset) {
                $record['preset'] = $annotation->preset;
            }

            if ($annotation->interface) {
                $record['interface'] = $annotation->interface;
            }

            $result[] = $record;
        }

        return $result;
    }
}
