<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action;

use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;
use MJS\TopSort\Implementations\StringSort;
use XCart\Extender\Domain\ModuleDependenciesMap;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Model\Entity;

use function array_diff;
use function array_merge;
use function strcmp;
use function usort;

class MixinSorter
{
    /**
     * @var ModuleDependenciesMap
     */
    private ModuleDependenciesMap $moduleDependenciesMap;

    /**
     * @var ReflectorInterface
     */
    private ReflectorInterface $reflector;

    /**
     * @param ModuleDependenciesMap $moduleDependenciesMap
     * @param ReflectorInterface    $reflector
     */
    public function __construct(
        ModuleDependenciesMap $moduleDependenciesMap,
        ReflectorInterface $reflector
    ) {
        $this->moduleDependenciesMap = $moduleDependenciesMap;
        $this->reflector             = $reflector;
    }

    /**
     * @param Entity[] $mixins
     *
     * @return Entity[]
     *
     * @throws EntityException
     * @throws LogicException
     */
    public function sort(array $mixins): array
    {
        if (empty($mixins)) {
            return [];
        }

        usort($mixins, static function (Entity $a, Entity $b) {
            return strcmp($b->getFqn(), $a->getFqn());
        });

        $mixinsMap          = [];
        $modules            = [];
        $moduleDependencies = [];

        foreach ($mixins as $mixin) {
            $mixinsMap[$mixin->getFqn()] = $mixin;

            $module = $this->reflector->getModule($mixin);

            // group mixins by module
            if (!isset($modules[$module])) {
                $modules[$module] = [];
            }
            $modules[$module][] = $mixin->getFqn();

            // find dependencies for each module
            // (list of mixins that should go before module)
            foreach ($this->reflector->getBeforeModules($mixin) as $beforeModule) {
                if (!isset($moduleDependencies[$beforeModule])) {
                    $moduleDependencies[$beforeModule] = [];
                }
                $moduleDependencies[$beforeModule][] = $mixin->getFqn();
            }
        }

        $sorter = new StringSort();

        foreach ($mixins as $mixin) {
            $module = $this->reflector->getModule($mixin);

            // merge mixin dependencies with mixin module dependencies
            $mixinDependencies = array_merge(
                $this->reflector->getAfterModules($mixin),
                // remove modules that should go after current mixin
                array_diff(
                    $this->moduleDependenciesMap[$module] ?? [],
                    $this->reflector->getBeforeModules($mixin)
                )
            );

            $dependencies = [];

            foreach ($mixinDependencies as $mixinDependency) {
                // get current mixin dependencies
                if (isset($modules[$mixinDependency])) {
                    $dependencies[] = $modules[$mixinDependency];
                }
            }

            // get current mixin module dependencies
            if (isset($moduleDependencies[$module])) {
                $dependencies[] = $moduleDependencies[$module];
            }

            $dependencies = array_diff($dependencies ? array_merge(...$dependencies) : [], [$mixin->getFqn()]);

            $sorter->add($mixin->getFqn(), $dependencies);
        }

        try {
            $sorted = $sorter->sort();
        } catch (CircularDependencyException | ElementNotFoundException $exception) {
            throw LogicException::fromMixinSorting($exception);
        }

        $result = [];
        foreach ($sorted as $fqn) {
            $result[] = $mixinsMap[$fqn];
        }

        return $result;
    }
}
