<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action;

use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\EntityFactoryInterface;
use XCart\Extender\Model\CheckMixinEvent;
use XCart\Extender\Model\Entity;

class MixinLookup implements MixinLookupInterface
{
    /**
     * @var SourceMapInterface
     */
    private SourceMapInterface $sourceMap;

    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var ReflectorInterface
     */
    private ReflectorInterface $reflector;

    /**
     * @var MixinSorter
     */
    private MixinSorter $mixinSorter;

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $eventDispatcher;

    /**
     * @var array|null
     */
    private ?array $cache = null;

    /**
     * @param SourceMapInterface       $sourceMap
     * @param EntityFactoryInterface   $entityFactory
     * @param ReflectorInterface       $reflector
     * @param MixinSorter              $mixinSorter
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        SourceMapInterface $sourceMap,
        EntityFactoryInterface $entityFactory,
        ReflectorInterface $reflector,
        MixinSorter $mixinSorter,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->sourceMap       = $sourceMap;
        $this->entityFactory   = $entityFactory;
        $this->reflector       = $reflector;
        $this->mixinSorter     = $mixinSorter;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param Entity $entity
     *
     * @return array|Entity[]
     * @throws EntityException
     * @throws LogicException
     */
    public function getMixins(Entity $entity): array
    {
        $this->warmup();

        $mixins = $this->cache[$entity->getFqn()] ?? [];

        if ($mixins && $this->reflector->isLocked($entity)) {
            throw EntityException::fromLockedEntityWithMixins($entity, $mixins);
        }

        return $this->mixinSorter->sort($mixins);
    }

    /**
     * @throws EntityException
     * @throws LogicException
     */
    public function warmup(): void
    {
        if ($this->cache === null) {
            foreach ($this->sourceMap->getModuleFiles() as $file) {
                $fqn = $this->sourceMap->suggestFqnByPath($file);
                if (empty($fqn)) {
                    continue;
                }

                $entity = $this->entityFactory->build($fqn, $file);

                if (!$this->reflector->isFQNEqualToPath($entity)) {
                    throw EntityException::fromFQNToPathEqualityCheck($entity);
                }

                if ($this->reflector->isMixin($entity)) {
                    $event = new CheckMixinEvent($entity);
                    $this->eventDispatcher->dispatch($event, 'check-mixin');

                    if ($event->getResult() !== false) {
                        $this->cache[$this->reflector->getParent($entity)][] = $entity;
                    }
                }
            }
        }
    }
}
