<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action\ClassCacheBuilder;

use XCart\Extender\Action\ReflectorInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\CodeFileFactory;
use XCart\Extender\Factory\EntityFactoryInterface;
use XCart\Extender\Model\Entity;

use function is_readable;

class EntityBuilder
{
    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var CodeFileFactory
     */
    private CodeFileFactory $codeFileFactory;

    /**
     * @var ReflectorInterface
     */
    private ReflectorInterface $reflector;

    /**
     * @var ChainBuilderInterface
     */
    private ChainBuilderInterface $chainBuilder;

    /**
     * @param EntityFactoryInterface $entityFactory
     * @param CodeFileFactory        $codeFileFactory
     * @param ReflectorInterface     $reflector
     * @param ChainBuilderInterface  $chainBuilder
     */
    public function __construct(
        EntityFactoryInterface $entityFactory,
        CodeFileFactory $codeFileFactory,
        ReflectorInterface $reflector,
        ChainBuilderInterface $chainBuilder
    ) {
        $this->entityFactory   = $entityFactory;
        $this->codeFileFactory = $codeFileFactory;
        $this->reflector       = $reflector;
        $this->chainBuilder    = $chainBuilder;
    }

    /**
     * @param Entity $entity
     *
     * @return string
     * @throws EntityException
     * @throws LogicException
     */
    public function buildByEntity(Entity $entity): string
    {
        if (!is_readable($entity->getSourcePath())) {
            throw EntityException::fromReadSource($entity);
        }

        if (!$this->reflector->isFQNEqualToPath($entity)) {
            throw EntityException::fromFQNToPathEqualityCheck($entity);
        }

        if (!$this->reflector->isClass($entity)) {
            $this->codeFileFactory->buildUnaltered($entity);

            return $entity->getTargetStream();
        }

        if (!$this->reflector->isCompatible($entity)) {
            return '';
        }

        $fqn = $entity->getFqn();

        if ($this->reflector->isMixin($entity)) {
            $entity = $this->entityFactory->build($this->reflector->getParent($entity));
        }

        $chain = $this->chainBuilder->buildChain($fqn, $entity);

        return $chain[$fqn] ?? '';
    }
}
