<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action\ClassCacheBuilder;

use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use XCart\Extender\Action\MixinLookupInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\CodeFileFactory;
use XCart\Extender\Factory\EntityFactoryInterface;
use XCart\Extender\Model\Entity;
use XCart\Extender\Model\EntityEvent;

class ChainBuilder implements ChainBuilderInterface
{
    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var CodeFileFactory
     */
    private CodeFileFactory $codeFileFactory;

    /**
     * @var MixinLookupInterface
     */
    private MixinLookupInterface $mixinLookup;

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $eventDispatcher;

    /**
     * @param EntityFactoryInterface   $entityFactory
     * @param CodeFileFactory          $codeFileFactory
     * @param MixinLookupInterface     $mixinLookup
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        EntityFactoryInterface $entityFactory,
        CodeFileFactory $codeFileFactory,
        MixinLookupInterface $mixinLookup,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->entityFactory   = $entityFactory;
        $this->codeFileFactory = $codeFileFactory;
        $this->mixinLookup     = $mixinLookup;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param string $fqn
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     * @throws LogicException
     */
    public function buildChain(string $fqn, Entity $entity): array
    {
        $chainEvent = new EntityEvent($entity);
        $this->eventDispatcher->dispatch($chainEvent, 'build-chain');

        if (($result = $chainEvent->getResult()) !== null) {
            return $result;
        }

        $mixins = $this->mixinLookup->getMixins($entity);
        if (!$mixins) {
            $this->codeFileFactory->buildUnaltered($entity);

            return [$entity->getFqn() => $entity->getTargetStream()];
        }

        $result = [];

        $ancestor = $this->entityFactory->buildAncestor($entity);

        $this->codeFileFactory->buildAncestorFromSource($ancestor);
        $result[$ancestor->getFqn()] = $ancestor->getTargetStream();

        $parentFqn = $ancestor->getFqn();
        foreach ($mixins as $mixin) {
            $mixin->setAncestorFqn($ancestor->getFqn());

            $this->codeFileFactory->addParentMutation($mixin, $parentFqn);
            $this->codeFileFactory->buildMixinFromSource($mixin);

            $result[$mixin->getFqn()] = $mixin->getTargetStream();

            $parentFqn = $mixin->getFqn();
        }

        $descendant = $this->entityFactory->buildDescendant($entity);
        $descendant->setAncestorFqn($ancestor->getFqn());

        $this->codeFileFactory->addParentMutation($descendant, $parentFqn);
        $this->codeFileFactory->buildDescendantFromScratch($descendant);

        $result[$descendant->getFqn()] = $descendant->getTargetStream();

        return $result;
    }
}
