<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action\ClassCacheBuilder;

use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetStreamMap;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;

class BuilderCached implements BuilderInterface
{
    /**
     * @var SourceMapInterface
     */
    private SourceMapInterface $sourceMap;

    /**
     * @var TargetStreamMap
     */
    private TargetStreamMap $targetStreamMap;

    /**
     * @var BuilderInterface
     */
    private BuilderInterface $builder;

    /**
     * @param SourceMapInterface $sourceMap
     * @param TargetStreamMap    $targetStreamMap
     * @param BuilderInterface   $builder
     */
    public function __construct(
        SourceMapInterface $sourceMap,
        TargetStreamMap $targetStreamMap,
        BuilderInterface $builder
    ) {
        $this->sourceMap       = $sourceMap;
        $this->targetStreamMap = $targetStreamMap;
        $this->builder         = $builder;
    }

    /**
     * @param string $fqn
     * @param string $sourcePath
     *
     * @return string
     * @throws EntityException
     * @throws LogicException
     */
    public function buildByFQN(string $fqn, string $sourcePath = ''): string
    {
        if (!isset($this->targetStreamMap[$fqn])) {
            $this->targetStreamMap[$fqn] = $this->builder->buildByFQN($fqn, $sourcePath);
        }

        return $this->targetStreamMap[$fqn];
    }

    /**
     * @param string $sourcePath
     *
     * @return string
     * @throws EntityException
     * @throws LogicException
     */
    public function buildByPath(string $sourcePath): string
    {
        $fqn = $this->sourceMap->suggestFqnByPath($sourcePath);

        return $fqn ? $this->buildByFQN($fqn, $sourcePath) : '';
    }
}
