<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action\ClassCacheBuilder;

use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\EntityFactoryInterface;

class Builder implements BuilderInterface
{
    /**
     * @var SourceMapInterface
     */
    private SourceMapInterface $sourceMap;

    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var EntityBuilder
     */
    private EntityBuilder $entityBuilder;

    /**
     * @param SourceMapInterface     $sourceMap
     * @param EntityFactoryInterface $entityFactory
     * @param EntityBuilder          $entityBuilder
     */
    public function __construct(
        SourceMapInterface $sourceMap,
        EntityFactoryInterface $entityFactory,
        EntityBuilder $entityBuilder
    ) {
        $this->sourceMap     = $sourceMap;
        $this->entityFactory = $entityFactory;
        $this->entityBuilder = $entityBuilder;
    }

    /**
     * @param string $fqn
     * @param string $sourcePath
     *
     * @return string
     * @throws EntityException
     * @throws LogicException
     */
    public function buildByFQN(string $fqn, string $sourcePath = ''): string
    {
        $entity = $this->entityFactory->build($fqn, $sourcePath);

        return $this->entityBuilder->buildByEntity($entity);
    }

    /**
     * @param string $sourcePath
     *
     * @return string
     * @throws EntityException
     * @throws LogicException
     */
    public function buildByPath(string $sourcePath): string
    {
        $fqn = $this->sourceMap->suggestFqnByPath($sourcePath);

        return $fqn ? $this->buildByFQN($fqn, $sourcePath) : '';
    }
}
