<?php

declare(strict_types=1);

namespace XCartMarketplace\Connector\Request;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Constraints as Assert;

class GetDataset extends ARequest
{
    protected string $target = 'get_dataset';

    protected static $allowedTargets = [
        'check_for_updates',
        'get_cores',
        'get_addons',
        'check_addon_key',
        'get_all_tags',
        'is_landing_available',
        'get_waves',
        'get_sections',
        'get_payment_methods',
        'get_editions',
    ];

    protected $requests = [];

    /**
     * Flag: true to use cache to store response
     *
     * @var bool
     */
    protected bool $useCache = false;

    /**
     * @return array
     */
    public static function getAllowedTargets(): array
    {
        return static::$allowedTargets;
    }

    /**
     * Add request to requests set
     * 
     * @param ARequest $request
     */
    public function addRequest(ARequest $request)
    {
        $this->requests[$request->getTarget()] = $request;
    }

    /**
     * Get request parameters
     *
     * @return array
     */
    public function getParams(): array
    {
        $params = [];

        foreach ($this->requests as $requestType => $request) {
            $params[$requestType] = $request->getParams() ?: [0];
        }

        return ['querySets' => $params];
    }

    /**
     * Return number of added requests
     * 
     * @return int
     */
    public function countRequests(): int
    {
        return count($this->requests);
    }

    /**
     * Return true if requests set contains specific request type
     * 
     * @param string $requestType
     * @return bool
     */
    public function contains(string $requestType): bool
    {
        return in_array($requestType, array_keys($this->requests));
    }

    /**
     * Format error response
     * 
     * @param int    $code    Error code
     * @param string $message Error message
     * @return array
     */
    public function formatErrorResponse(int $code, string $message): array
    {
        $result = [];

        foreach ($this->requests as $requestType => $request) {
            $result[$requestType] = $request->formatErrorResponse($code, $message);
        }

        return $result;
    }

    /**
     * @return Constraint|null
     */
    public static function getResponseValidationSchema(): ?Constraint
    {
        return null;
    }
}
