<?php

declare(strict_types=1);

namespace XCartMarketplace\Connector;

use Symfony\Component\Validator\Constraints as Assert;

class Config
{
    /**
     * @var array
     */
    private array $options;

    /**
     * List of service options names, i.e. options which should not be used as a common request parameters
     *
     * @var string[]
     */
    private array $serviceOptionsNames = ['url'];

    /**
     * @param array $options
     */
    public function __construct(array $options = [])
    {
        $this->options = $options;
    }

    /**
     * Validate config options agains the validation schema and return the list of errors
     *
     * @param Validator $validator Validator object
     *
     * @return string[]
     */
    public function validate(Validator $validator): array
    {
        return $validator->validate($this->options, $this->getValidationSchema(), 'Config');
    }

    /**
     * Get endpoint URL
     *
     * @return string
     */
    public function getUrl(): string
    {
        return $this->getOption('url');
    }

    /**
     * Return common request parameters
     *
     * @return array
     */
    public function getCommonParams(): array
    {
        return array_diff_key($this->options, array_flip($this->serviceOptionsNames));
    }

    /**
     * Get validation schema to validate config parameters
     *
     * @return Assert\Collection
     */
    public static function getValidationSchema(): Assert\Collection
    {
        return new Assert\Collection([
            'url'                => [
                new Assert\NotBlank(),
                new Assert\Url(),
            ],
            'email'              => [
                new Assert\NotBlank(),
                new Assert\Email(),
            ],
            'shopID'             => new Assert\NotBlank(),
            'shopDomain'         => new Assert\NotBlank(),
            'shopURL'            => [
                new Assert\NotBlank(),
                new Assert\Url(),
            ],
            'currentCoreVersion' => new Assert\Collection([
                'major' => new Assert\Regex(['pattern' => '/^\d+\.\d+$/']),
                'minor' => new Assert\PositiveOrZero(),
                'build' => new Assert\PositiveOrZero(),
            ]),
            'shopCountryCode'    => new Assert\Country(),
            'trial'              => new Assert\Type(['type' => 'bool']),
            'cloud'              => new Assert\Type(['type' => 'bool']),
            'xcn_license_key'    => new Assert\Optional(
                new Assert\Type('string'),
            ),
            'customer_id'        => new Assert\Optional(
                new Assert\Type('string'),
            ),
            'affiliateId'        => new Assert\Optional(
                new Assert\Type('string'),
            ),
            'installation_lng'   => new Assert\Language(),
        ]);
    }

    /**
     * Get option value by its name
     *
     * @param string $name Option name
     *
     * @return mixed
     */
    public function getOption(string $name)
    {
        return $this->options[$name] ?? null;
    }

    /**
     * @param string $name
     * @param string $value
     */
    public function setOption(string $name, string $value): void
    {
        $this->options[$name] = $value;
    }
}
