<?php

declare(strict_types=1);

namespace XCartMarketplace\Connector;

use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use Psr\Cache\CacheItemPoolInterface as CacheInterface;
use XCartMarketplace\Connector\Request\ARequest;
use XCartMarketplace\Connector\Request\GetDataset;
use XCartMarketplace\Connector\Exceptions;

class Client
{
    /**
     * Marketplace Store API version
     */
    public const VERSION_API = '2.7';

    /**
     * Connector configuration
     *
     * @var Config
     */
    private Config $config;

    /**
     * @var Validator
     */
    private Validator $validator;

    /**
     * @var CacheInterface
     */
    private CacheInterface $cache;

    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * Errors list
     *
     * @var string[]
     */
    private array $errors = [];

    /**
     * Queued requests
     *
     * @var array
     */
    private array $requests = [];

    /**
     * Response data
     *
     * @var array
     */
    private array $response = [];

    /**
     * GetDataset request instance
     *
     * @var GetDataset
     */
    private $datasetRequest;

    /**
     * Hash of cache keys of the requests groupped to $datasetRequest request
     *
     * @var array
     */
    private array $datasetRequestHash = [];

    /**
     * Constructor
     */
    public function __construct(Config $config, CacheInterface $cache, LoggerInterface $logger = null)
    {
        $this->config    = $config;
        $this->validator = new Validator();
        $this->cache     = $cache;
        $this->logger    = $logger ?? new NullLogger();
    }

    /**
     * Get errors list
     *
     * @return array
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * Add request to the requests queue and return true on success of false if request data validation is failed
     *
     * @param string $target Request target
     * @param array  $params Request parameters
     * @param bool   $ignoreCache Flag: true - ignore cache and force sending request
     * @param array  $serviceOptions Additional (service) options, e.g. 'range' for get_addon_pack/get_core_pack
     * @throws Exceptions\ClientException
     */
    public function addRequest(string $target, array $params = [], bool $ignoreCache = false, array $serviceOptions = [], int $timeout = 15, int $connectTimeout = 5): void
    {
        if ($target == 'get_dataset') {
            // It's not allowed to create get_dataset requests directly
            return;
        }

        $class = '\\XCartMarketplace\\Connector\\Request\\' . implode('', array_map('ucfirst', explode('_', strval($target))));
        if (!class_exists($class)) {
            throw new Exceptions\ClientException(sprintf('XCartMarketplaceConnector: Unexpected request target "%s"', strval($target)));
        }

        $request = new $class($params, $ignoreCache, $serviceOptions, $timeout, $connectTimeout);

        $this->errors = array_merge(
            $this->errors,
            $request->validate($this->validator)
        );

        if (empty($this->errors)) {
            $this->requests[$request->getTarget()] = $request;
        }
    }

    /**
     * Reset requests queue
     */
    public function reset()
    {
        $this->requests = [];
        $this->errors = [];
    }

    /**
     * Do queued requests and return marketplace response
     *
     * @return array
     * @throws Exceptions\RequestValidationException
     */
    public function getData(): array
    {
        $this->errors = array_merge(
            $this->errors,
            $this->config->validate($this->validator)
        );

        if ($this->errors) {
            foreach ($this->errors as $e) {
                $this->logger->error($e);
            }
            throw new Exceptions\RequestValidationException($this->errors[0]);
        }

        $this->response = [];
        $this->datasetRequest = null;
        $this->datasetRequestHash = [];

        $this->preprocessRequestsQueue();

        $this->doRequests();

        $this->reset();

        return $this->response;
    }

    /**
     * Preprocess requests queue:
     *   - Prepare responses from the cache
     *   - Prepare GetDataset request
     */
    protected function preprocessRequestsQueue()
    {
        foreach ($this->requests as $requestType => $request) {
            $params = $this->prepareRequestParams($request);

            $cacheKey = $request->getCacheKey($params);

            if (!$request->isIgnoreCache()) {
                $cacheItem = $this->cache->getItem($cacheKey);
                $cachedResponse = $cacheItem->get();
                if ($cachedResponse) {
                    $this->response[$requestType] = $cachedResponse;
                    $this->logger->info('Got cached response', ['type' => $requestType, 'response' => $cachedResponse]);
                    unset($this->requests[$requestType]);
                    continue;
                }
            }

            if ($request->isAllowedToSendInDataset()) {
                $this->datasetRequest = $this->datasetRequest ?: new GetDataset([], true);
                $this->datasetRequest->addRequest($request);
                $this->datasetRequestHash[$requestType] = $cacheKey;
            }
        }

        if ($this->datasetRequest && $this->datasetRequest->countRequests() > 1) {
            // We use get_dataset only if it contains more than 2 requests
            $this->requests[$this->datasetRequest->getTarget()] = $this->datasetRequest;
        } else {
            $this->datasetRequest = null;
            $this->datasetRequestHash = [];
        }
    }

    /**
     * Do queued requests and prepare response
     */
    protected function doRequests()
    {
        foreach ($this->requests as $requestType => $request) {

            if ($requestType != 'get_dataset' && $this->datasetRequest && $this->datasetRequest->contains($requestType)) {
                // Skip request as it will be processed as part of get_dataset
                continue;
            }

            $params = $this->prepareRequestParams($request);

            $url = $this->config->getUrl() . $request->getTarget();

            $this->logger->info('Send request', ['type' => $requestType, 'url' => $url, 'params' => $params]);

            $response = Sender::send($request->getMethod(), $url, $params, $request->getHeaders(), $request->getTimeout(), $request->getConnectTimeout());

            if ($error = $response->getError()) {
                $this->logger->error($error);
                $this->prepareErrorResponse($request, -1, $error);
                continue;
            }

            if ($errors = $response->validate($request, $this->validator)) {
                $this->logger->info('Response', $response->getResponse());
                foreach ($errors as $e) {
                    $this->logger->error($e);
                }
                $this->prepareErrorResponse($request, -2, $errors[0]);
                continue;
            }

            $this->logger->info('Got response', ['request type' => $requestType, 'response' => $this->prepareLogData($response->getResponse())]);

            if ($response) {
                if ($requestType == 'get_dataset') {
                    foreach ($response->getResponse() as $rType => $resp) {
                        $this->response[$rType] = $resp;
                        $this->cacheResponse($this->requests[$rType], $this->datasetRequestHash[$rType], $resp);
                    }
                    continue;
                }

                $this->cacheResponse($request, $request->getCacheKey($params), $response->getResponse());

                $this->response[$requestType] = $response->getResponse();
            }
        }
    }

    /**
     * Prepare error response for request
     * 
     * @param ARequest $request
     * @param int      $code
     * @param string   $error
     */
    protected function prepareErrorResponse(ARequest $request, int $code, string $error)
    {
        if ($request->getTarget() == 'get_dataset') {
            $this->response = array_merge($this->response, $request->formatErrorResponse($code, $error));
        } else {
            $this->response[$request->getTarget()] = $request->formatErrorResponse($code, $error);
        }
    }

    /**
     * Save response to the cache
     * 
     * @param ARequest $request
     * @param string   $cacheKey
     * @param array $response
     */
    protected function cacheResponse(ARequest $request, string $cacheKey, ?array $response)
    {
        if ($request->isUseCache()) {
            $cacheItem = $this->cache->getItem($cacheKey);
            $cacheItem->set($response);
            $cacheItem->expiresAfter($request->getCacheTTL());
            $this->cache->save($cacheItem);
        }
    }

    /**
     * Return request parameters merged with common parameters
     *
     * @param ARequest $request
     *
     * @return array
     */
    protected function prepareRequestParams(ARequest $request): array
    {
        return array_merge(
            ['versionAPI' => static::VERSION_API],
            $this->config->getCommonParams(),
            $request->getParams()
        );
    }

    /**
     * Prepare data for logging
     * 
     * @param array $data
     * @return array
     */
    protected function prepareLogData($data): array
    {
        if (!empty($data['body'])) {
            // Replace binary data to special marker
            $data['body'] = sprintf('[DATA:%db]', strlen((string)$data['body']));
        }

        return $data;
    }
}
