<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

/** @noinspection PhpInternalEntityUsedInspection */

declare(strict_types=1);

namespace XCart\Bundle\LogicBundle\DependencyInjection\Compiler\Parser;

use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocTagValueNode;
use PHPStan\PhpDocParser\Lexer\Lexer;
use PHPStan\PhpDocParser\Parser\ConstExprParser;
use PHPStan\PhpDocParser\Parser\PhpDocParser;
use PHPStan\PhpDocParser\Parser\TokenIterator;
use PHPStan\PhpDocParser\Parser\TypeParser;
use ReflectionClass;
use ReflectionException;
use Symfony\Component\PropertyInfo\PhpStan\NameScopeFactory;
use Symfony\Component\PropertyInfo\Type;
use Symfony\Component\PropertyInfo\Util\PhpStanTypeHelper;

class ActionParser
{
    private const ACTION_METHOD_NAME = 'run';

    private PhpDocParser $phpDocParser;

    private Lexer $lexer;

    private NameScopeFactory $nameScopeFactory;

    private PhpStanTypeHelper $phpStanTypeHelper;

    public function __construct()
    {
        $constExprParser = new ConstExprParser();
        $typeParser = new TypeParser($constExprParser);

        $this->phpDocParser = new PhpDocParser($typeParser, $constExprParser);
        $this->lexer = new Lexer();

        $this->nameScopeFactory = new NameScopeFactory();
        $this->phpStanTypeHelper = new PhpStanTypeHelper();
    }

    /**
     * @return string[]
     */
    public function getParamClasses(string $className): array
    {
        $tokens = $this->parse($className);
        if (!$tokens) {
            return [];
        }

        $params = $tokens->getTagsByName('@param');
        if (count($params) === 0) {
            return [];
        }

        $types = $this->parseTypes($className, $params[0]->value);
        if (count($types) === 0) {
            return [];
        }

        $classes = [];
        foreach ($types as $type) {
            if ($type->getClassName()) {
                $classes[] = $type->getClassName();
            }
        }

        return $classes;
    }

    /**
     * @return string[]
     */
    public function getReturnClasses(string $className): array
    {
        $tokens = $this->parse($className);
        if (!$tokens) {
            return [];
        }

        $params = $tokens->getTagsByName('@return');
        if (count($params) === 0) {
            return [];
        }

        $param = reset($params);

        $types = $this->parseTypes($className, $param->value);
        if (count($types) === 0) {
            return [];
        }

        $classes = [];
        foreach ($types as $type) {
            if ($type->getClassName()) {
                $classes[] = $type->getClassName();
            }
        }

        return $classes;
    }

    public function parse(string $className): ?PhpDocNode
    {
        try {
            $reflectionClass = new ReflectionClass($className);
        } catch (ReflectionException) {
            return null;
        }

        if (!$reflectionClass->hasMethod(self::ACTION_METHOD_NAME)) {
            return null;
        }

        $reflectionMethod = $reflectionClass->getMethod(self::ACTION_METHOD_NAME);

        $tokens = $this->lexer->tokenize($reflectionMethod->getDocComment());

        return $this->phpDocParser->parse(new TokenIterator($tokens));
    }

    /**
     * @return Type[]
     */
    public function parseTypes(string $className, PhpDocTagValueNode $value): array
    {
        $nameScope = $this->nameScopeFactory->create($className, $className);

        return $this->phpStanTypeHelper->getTypes($value, $nameScope);
    }
}
