<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\LogicBundle\Decorator\Cacher;

use Psr\Cache\CacheItemInterface;
use Psr\Cache\InvalidArgumentException;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Symfony\Contracts\Cache\CacheInterface;
use Throwable;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\Assembler\Response\DefaultFailedResponseAssemblerInterface;
use XCart\Bundle\LogicBundle\Decorator\Cacher\Assembler\CacheItemKey\CacheItemKeyAssemblerInterface;
use XCart\Bundle\LogicBundle\Decorator\Cacher\Preprocessor\CacheItem\CacheItemPreprocessorInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;

class Cacher implements ActionInterface
{
    use LoggerAwareTrait;

    private const PREFIX = 'logic_cache';

    public function __construct(
        private ActionInterface $inner,
        private CacheInterface $cache,
        private CacheItemKeyAssemblerInterface $cacheItemKeyAssembler,
        private CacheItemPreprocessorInterface $cacheItemPreprocessor,
        private DefaultFailedResponseAssemblerInterface $failedResponseAssembler,
        LoggerInterface $logger,
    ) {
        $this->setLogger($logger);
    }

    public function run(RequestInterface $request): ResponseInterface
    {
        try {
            $key = $this->cacheItemKeyAssembler->assemble($request);
        } catch (Throwable $e) {
            $this->logger->error('Cannot assemble cache item key', ['exception' => $e, 'request' => $request]);

            return $this->inner->run($request);
        }

        $key = self::PREFIX . $key;

        try {
            return $this->cache->get(
                $key,
                function (CacheItemInterface $item) use ($request) {
                    $this->cacheItemPreprocessor->prepare($item);

                    return $this->inner->run($request);
                },
            );
        } catch (InvalidArgumentException $e) {
            $this->logger->error('Cannot get cache item (invalid cache arguments)', ['exception' => $e, 'request' => $request]);

            return $this->failedResponseAssembler->assemble();
        }
    }
}
