<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service;

use JMS\Serializer\SerializerInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\APIEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DTO;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\EntityInterface;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicRequestEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicResponseEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Transformer;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\TransformersDataProvider as DataProvider;

class Register
{
    public function __construct(private NameResolverInterface $nameResolver)
    {
    }

    public function registerTransformers(ContainerBuilder $container, DataProvider $dataProvider): void
    {
        /** @var Transformer $entity */
        foreach ($dataProvider as $entity) {
            if (!$entity->pattern->isPreDefined()) {
                $this->registerTransformer($container, $entity);
            }

            $this->registerTransformerInterface($container, $entity);
        }
    }

    public function registerFactories(ContainerBuilder $container, $dataProvider): void
    {
        /** @var APIEntity|DoctrineEntity|LogicEntity|LogicRequestEntity|LogicResponseEntity|DTO $entity */
        foreach ($dataProvider as $entity) {
            if ($entity instanceof DTO && $entity->trait) {
                continue;
            }

            $this->registerDTOFactory($container, $entity);
        }
    }

    private function registerTransformer(ContainerBuilder $container, Transformer $entity): void
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);

        $id = $className;
        $definition = (new Definition($className))
            ->setAutowired(true)
            ->setAutoconfigured(true);

        if (
            $entity->type
            && (
                $entity->type->isLogicResponseItemPayload()
                || $entity->type->isLogicResponseCollectionPayload()
                || $entity->type->isLogicRequestItemPayload()
            )
        ) {
            $intermediateTransformer = $this->nameResolver
                ->buildTransformerInterfaceName($entity->intermediateTransformer);
            $factory = $this->nameResolver->buildFactoryInterfaceName($entity->to);
            $definition->setArgument('$transformer', new Reference($intermediateTransformer));
            if ($entity->toEntity) {
                $definition->setArgument('$factory', new Reference($factory));
            }
        } elseif ($entity->pattern->isDirect() || $entity->pattern->isDirectTo()) {
            $definition->setArgument(
                '$factory',
                new Reference($this->nameResolver->buildFactoryInterfaceName($entity->to)),
            );
        }

        $container->setDefinition($id, $definition);
    }

    private function registerTransformerInterface(ContainerBuilder $container, Transformer $entity): void
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);

        $id = $className;
        $container->setAlias($this->nameResolver->buildTransformerInterfaceName($entity), $id);
    }

    private function registerDTOFactory(ContainerBuilder $container, EntityInterface $entity): void
    {
        $className = $this->nameResolver->buildFactoryClassName($entity->className);

        $id = $className;
        $definition = (new Definition($className));

        $container->setDefinition($id, $definition);
        $container->setAlias($this->nameResolver->buildFactoryInterfaceName($entity->className), $id);
    }
}
