<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\PostProcessor;

use RuntimeException;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DoctrineEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicRequestEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicResponseEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Transformer;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTODataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicRequestEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicResponseEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\TransformersDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicResponseType;
use XCart\Bundle\DTOGeneratorBundle\Enum\TransformerPattern;
use XCart\Bundle\DTOGeneratorBundle\Enum\TransformerType;

class TransformerPostProcessor
{
    public function __construct(
        private LogicEntitiesDataProvider $logicEntitiesDataProvider,
        private LogicRequestEntitiesDataProvider $logicRequestEntitiesDataProvider,
        private LogicResponseEntitiesDataProvider $logicResponseEntitiesDataProvider,
        private DoctrineEntitiesDataProvider $doctrineEntitiesDataProvider,
        private DTODataProvider $DTODataProvider,
    ) {
    }

    public function normalize(DataProvider $dataProvider): void
    {
        /** @var Transformer $entity */
        foreach ($dataProvider as $entity) {
            if (!$entity->toEntity) {
                $this->assignToEntities($entity);
            }
            if (!$entity->fromEntity) {
                $this->assignFromEntities($entity);
            }
        }

        /** @var Transformer $entity */
        foreach ($dataProvider as $entity) {
            if ($entity->type?->getValue() === TransformerType::LOGIC_RESPONSE_ITEM_PAYLOAD) {
                $entity->pattern = TransformerPattern::SERIALIZATION();
                $this->processLogicItemResponse($entity);
            }

            if ($entity->type?->getValue() === TransformerType::LOGIC_RESPONSE_COLLECTION_PAYLOAD) {
                $entity->pattern = TransformerPattern::SERIALIZATION();
                $this->processLogicCollectionResponse($entity);
            }

            if ($entity->type?->getValue() === TransformerType::LOGIC_REQUEST_ITEM_PAYLOAD) {
                $entity->pattern = TransformerPattern::SERIALIZATION();
                $this->processLogicItemRequest($entity);
            }
        }

        /** @var Transformer $entity */
        foreach ($dataProvider as $entity) {
            if ($entity->type?->getValue() === TransformerType::LOGIC_RESPONSE_ITEM_PAYLOAD) {
                $this->detectInternalTransformerForLogicItemResponse($entity, $dataProvider);
            }

            if ($entity->type?->getValue() === TransformerType::LOGIC_RESPONSE_COLLECTION_PAYLOAD) {
                $this->detectInternalTransformerForLogicCollectionResponse($entity, $dataProvider);
            }

            if ($entity->type?->getValue() === TransformerType::LOGIC_REQUEST_ITEM_PAYLOAD) {
                $this->detectInternalTransformerForLogicItemRequest($entity, $dataProvider);
            }
        }

        /** @var Transformer $entity */
        foreach ($dataProvider as $entity) {
            if (($entity->pattern->isDirect() || $entity->pattern->isDirectTo()) && !$entity->toEntity) {
                throw new \InvalidArgumentException(
                    sprintf(
                        'Transformer from %s to %s cannot be direct, because %s entity is not generated',
                        $entity->from,
                        $entity->to,
                        $entity->to,
                    )
                );
            }

            if (($entity->pattern->isDirect() || $entity->pattern->isDirectFrom()) && !$entity->fromEntity) {
                throw new \InvalidArgumentException(
                    sprintf(
                        'Transformer from %s to %s cannot be direct, because %s entity is not generated',
                        $entity->from,
                        $entity->to,
                        $entity->from,
                    )
                );
            }
        }
    }

    private function assignToEntities(Transformer $entity): void
    {
        if ($this->logicEntitiesDataProvider->hasClassName($entity->to)) {
            $entity->toEntity = $this->logicEntitiesDataProvider->getByClassName($entity->to);
        }

        if ($this->logicRequestEntitiesDataProvider->hasClassName($entity->to)) {
            $entity->toEntity = $this->logicRequestEntitiesDataProvider->getByClassName($entity->to);
        }

        if ($this->logicResponseEntitiesDataProvider->hasClassName($entity->to)) {
            $entity->toEntity = $this->logicResponseEntitiesDataProvider->getByClassName($entity->to);
        }

        if ($this->doctrineEntitiesDataProvider->hasClassName($entity->to)) {
            $entity->toEntity = $this->doctrineEntitiesDataProvider->getByClassName($entity->to);
        }

        if ($this->DTODataProvider->hasClassName($entity->to)) {
            $entity->toEntity = $this->DTODataProvider->getByClassName($entity->to);
        }
    }

    private function assignFromEntities(Transformer $entity): void
    {
        if ($this->logicEntitiesDataProvider->hasClassName($entity->from)) {
            $entity->fromEntity = $this->logicEntitiesDataProvider->getByClassName($entity->from);
        }

        if ($this->logicRequestEntitiesDataProvider->hasClassName($entity->from)) {
            $entity->fromEntity = $this->logicRequestEntitiesDataProvider->getByClassName($entity->from);
        }

        if ($this->logicResponseEntitiesDataProvider->hasClassName($entity->from)) {
            $entity->fromEntity = $this->logicResponseEntitiesDataProvider->getByClassName($entity->from);
        }

        if ($this->doctrineEntitiesDataProvider->hasClassName($entity->from)) {
            $entity->fromEntity = $this->doctrineEntitiesDataProvider->getByClassName($entity->from);
        }

        if ($this->DTODataProvider->hasClassName($entity->from)) {
            $entity->fromEntity = $this->DTODataProvider->getByClassName($entity->from);
        }
    }

    private function processLogicItemResponse(Transformer $entity): void
    {
        /** @var DoctrineEntity $doctrineEntity */
        foreach ($this->doctrineEntitiesDataProvider as $doctrineEntity) {
            if ($doctrineEntity->className === $entity->from) {
                $entity->fromEntity = $doctrineEntity;
            }
        }

        /** @var LogicResponseEntity $logicEntity */
        foreach ($this->logicResponseEntitiesDataProvider as $logicEntity) {
            if ($logicEntity->className === $entity->to) {
                $entity->toEntity = $logicEntity;
            }
        }

        if (!$entity->toEntity) {
            throw new RuntimeException(
                sprintf('Cannot find logic response "%s" for transformer from "%s"', $entity->to, $entity->from)
            );
        }
    }

    private function processLogicCollectionResponse(Transformer $entity): void
    {
        $this->processLogicItemResponse($entity);
    }

    private function processLogicItemRequest(Transformer $entity): void
    {
        /** @var LogicRequestEntity $logicRequestEntity */
        foreach ($this->logicRequestEntitiesDataProvider as $logicRequestEntity) {
            if ($logicRequestEntity->className === $entity->from) {
                $entity->fromEntity = $logicRequestEntity;
            }
        }

        if (!$entity->fromEntity) {
            throw new RuntimeException(
                sprintf('Cannot find logic entity "%s" for transformer to "%s"', $entity->from, $entity->to)
            );
        }

        /** @var DoctrineEntity $doctrineEntity */
        foreach ($this->doctrineEntitiesDataProvider as $doctrineEntity) {
            if ($doctrineEntity->className === $entity->to) {
                $entity->toEntity = $doctrineEntity;
            }
        }
    }

    private function detectInternalTransformerForLogicItemResponse(Transformer $entity, DataProvider $dataProvider): void
    {
        /** @var Transformer $subEntity */
        foreach ($dataProvider as $subEntity) {
            if ($subEntity->from === $entity->from) {
                $suffix = preg_replace('/(^|\\\\)New(\w+)$/Ss', '\1\2', preg_replace('/^.+\\\\Logic\\\\Entity\\\\/Ss', '', $subEntity->to));

                /** @var LogicEntity $logicEntity */
                foreach ($this->logicEntitiesDataProvider as $logicEntity) {
                    if (
                        $logicEntity->className === $subEntity->to
                        && str_contains($entity->to, $suffix)
                        && (
                            ($entity->toEntity->type?->getValue() === LogicResponseType::CREATE_ONE && $logicEntity->new)
                            || ($entity->toEntity->type?->getValue() !== LogicResponseType::CREATE_ONE && !$logicEntity->new)
                        )
                    ) {
                        $entity->intermediateTransformer = $subEntity;
                        $entity->intermediateEntity = $logicEntity;

                        return;
                    }
                }
            }
        }

        throw new RuntimeException(
            sprintf('Cannot find intermediate transformer for transformer from "%s" to "%s"', $entity->from, $entity->to)
        );
    }

    private function detectInternalTransformerForLogicCollectionResponse(Transformer $entity, DataProvider $dataProvider): void
    {
        $this->detectInternalTransformerForLogicItemResponse($entity, $dataProvider);
    }

    private function detectInternalTransformerForLogicItemRequest(Transformer $entity, DataProvider $dataProvider): void
    {
        /** @var Transformer $subEntity */
        foreach ($dataProvider as $subEntity) {
            if (
                $subEntity->to === $entity->to
                && $entity->fromEntity->entity === $subEntity->from
            ) {
                $suffix = preg_replace('/(^|\\\\)New(\w+)$/Ss', '\1\2', preg_replace('/^.+\\\\Logic\\\\Entity\\\\/Ss', '', $subEntity->to));

                /** @var LogicEntity $logicEntity */
                foreach ($this->logicEntitiesDataProvider as $logicEntity) {
                    if (
                        $logicEntity->className === $subEntity->from
                        && str_contains($entity->to, $suffix)
                        && (
                            ($entity->fromEntity->type?->getValue() === LogicRequestType::CREATE_ONE && $logicEntity->new)
                            || ($entity->fromEntity->type?->getValue() !== LogicRequestType::CREATE_ONE && !$logicEntity->new)
                        )
                    ) {
                        $entity->intermediateTransformer = $subEntity;
                        $entity->intermediateEntity = $logicEntity;

                        return;
                    }
                }
            }
        }

        throw new RuntimeException(
            sprintf('Cannot find intermediate transformer for transformer from "%s" to "%s"', $entity->from, $entity->to)
        );
    }
}
