<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\PostProcessor;

use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider as DataProvider;

class LogicEntityPostProcessor
{
    public function normalize(DataProvider $dataProvider): void
    {
        foreach ($dataProvider as $entity) {
            $this->sanitizeClassNames($entity);
            $this->moveClassNamesToUse($entity);
            $this->checkUniqueness($entity);
            $this->sortClassNames($entity);
        }
    }

    private function sanitizeClassNames(Entity $entity): void
    {
        foreach ($entity->interfaces as $k => $className) {
            $entity->interfaces[$k] = trim($className, '\\');
        }

        foreach ($entity->traits as $k => $className) {
            $entity->traits[$k] = trim($className, '\\');
        }
    }

    private function moveClassNamesToUse(Entity $entity): void
    {
        foreach ($entity->interfaces as $k => $className) {
            if (str_contains($className, '\\')) {
                $entity->uses[] = $className;
                $entity->interfaces[$k] = ClassNameAssembler::assembleShortName($className);
            }
        }

        foreach ($entity->traits as $k => $className) {
            if (str_contains($className, '\\')) {
                $entity->uses[] = $className;
                $entity->traits[$k] = ClassNameAssembler::assembleShortName($className);
            }
        }

        if ($entity->extends && str_contains($entity->extends, '\\')) {
            $entity->uses[] = $entity->extends;
            $entity->extends = ClassNameAssembler::assembleShortName($entity->extends);
        }

        foreach ($entity->fields as $field) {
            if (str_contains($field->type, '\\')) {
                $entity->uses[] = preg_replace('/\[]$/S', '', $field->type);
                $field->type = ClassNameAssembler::assembleShortName($field->type);
            }
        }
    }

    private function checkUniqueness(Entity $entity): void
    {
        $entity->uses = array_unique($entity->uses);
        $entity->interfaces = array_unique($entity->interfaces);
    }

    private function sortClassNames(Entity $entity): void
    {
        foreach ($entity->fields as $field) {
            $field->annotations = $field->annotations->sortByName();
        }

        sort($entity->uses);
        sort($entity->interfaces);
        $entity->annotations = $entity->annotations->sortByName();
    }
}
