<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Normalizer;

use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use XCart\Bundle\DTOGeneratorBundle\Collection\Annotations;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Field;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicResponseEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicResponseEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicResponseType;
use XCart\Bundle\DTOGeneratorBundle\Service\Normalizer\EntityProcessor\EntityProcessorInterface;
use XCart\Bundle\LogicBundle\DTO\Response\PayloadListOwnerResponseInterface;
use XCart\Bundle\LogicBundle\DTO\Response\PayloadOwnerResponseInterface;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;

/**
 * @property iterable<int,EntityProcessorInterface> $processors
 */
class LogicResponseEntityNormalizer implements NormalizerInterface
{
    public function __construct(
        private DataProvider $dataProvider,
        private iterable $processors,
    ) {
    }

    public function normalize(): void
    {
        foreach ($this->dataProvider as $entity) {
            $this->normalizeEntity($entity);
        }
    }

    private function normalizeEntity(Entity $entity): void
    {
        $entity->interfaces[] = ResponseInterface::class;

        switch ($entity->type?->getValue()) {
            case LogicResponseType::GET_ONE:
            case LogicResponseType::CREATE_ONE:
            case LogicResponseType::UPDATE_ONE:
                if (!$entity->entity) {
                    throw new InvalidConfigurationException(
                        sprintf('If logic response type is "%s" entity field must be filled', $entity->type)
                    );
                }
                $entity->fields['payload'] = new Field();
                $entity->fields['payload']->type = $entity->entity;
                $entity->fields['payload']->name = 'payload';
                $entity->fields['payload']->annotations = new Annotations();

                $entity->interfaces[] = PayloadOwnerResponseInterface::class;
                break;

            case LogicResponseType::GET_COLLECTION:
                if (!$entity->entity) {
                    throw new InvalidConfigurationException(
                        sprintf('If logic response type is "%s" entity field must be filled', $entity->type)
                    );
                }
                $entity->fields['payload'] = new Field();
                $entity->fields['payload']->type = $entity->entity . '[]';
                $entity->fields['payload']->name = 'payload';
                $entity->fields['payload']->annotations = new Annotations();
                $entity->fields['payload']->annotations
                    ->addAnnotationFromArray(['values' => 'array<' . $entity->entity . '>'], 'JMS\Type');

                $entity->interfaces[] = PayloadListOwnerResponseInterface::class;
                break;
        }

        foreach ($this->processors as $processor) {
            $processor->process($entity);
        }
    }
}
