<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Normalizer;

use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use XCart\Bundle\DTOGeneratorBundle\Collection\Annotations;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Field;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicRequestEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicRequestEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;
use XCart\Bundle\DTOGeneratorBundle\Service\Normalizer\EntityProcessor\EntityProcessorInterface;
use XCart\Bundle\LogicBundle\DTO\Request\IdBasedRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Request\Uuid4BasedRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Request\PayloadOwnerRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;

/**
 * @property iterable<int,EntityProcessorInterface> $processors
 */
class LogicRequestEntityNormalizer implements NormalizerInterface
{
    public function __construct(
        private LogicRequestEntitiesDataProvider $dataProvider,
        private LogicEntitiesDataProvider $logicEntitiesDataProvider,
        private iterable $processors,
    ) {
    }

    public function normalize(): void
    {
        foreach ($this->dataProvider as $entity) {
            $this->normalizeEntity($entity);
        }
    }

    private function normalizeEntity(Entity $entity): void
    {
        $entity->interfaces[] = RequestInterface::class;

        /** @var LogicEntity $logicEntity */
        $linkedEntity = null;
        if ($entity->entity) {
            /** @var LogicEntity $logicEntity */
            foreach ($this->logicEntitiesDataProvider as $logicEntity) {
                if ($logicEntity->className === $entity->entity) {
                    $linkedEntity = $logicEntity;
                    break;
                }
            }
        }

        if (!$linkedEntity) {
            throw new InvalidConfigurationException(
                sprintf('If logic request type is "%s" entity field must be filled', $entity->type)
            );
        }

        switch ($entity->type?->getValue()) {
            case LogicRequestType::GET_ONE:
            case LogicRequestType::DELETE_ONE:
                if ($linkedEntity->idType === LogicEntity::ID_TYPE_INT) {
                    $entity->fields['id'] = new Field();
                    $entity->fields['id']->type = 'integer';
                    $entity->fields['id']->name = 'id';
                    $entity->fields['id']->annotations = new Annotations();
                    $entity->fields['id']->annotations->addAnnotationFromArray([], 'Assert\\Positive');

                    $entity->interfaces[] = IdBasedRequestInterface::class;

                } elseif ($linkedEntity->idType === LogicEntity::ID_TYPE_UUID4) {
                    $entity->fields['id'] = new Field();
                    $entity->fields['id']->type = 'string';
                    $entity->fields['id']->name = 'id';
                    $entity->fields['id']->annotations = new Annotations();
                    $entity->fields['id']->annotations->addAnnotationFromArray(['versions' => [4]], 'Assert\\Uuid');

                    $entity->interfaces[] = Uuid4BasedRequestInterface::class;
                }
                break;

            case LogicRequestType::CREATE_ONE:
            case LogicRequestType::UPDATE_ONE:
                $entity->fields['payload'] = new Field();
                $entity->fields['payload']->type = $entity->entity;
                $entity->fields['payload']->name = 'payload';
                $entity->fields['payload']->annotations = new Annotations();

                $entity->interfaces[] = PayloadOwnerRequestInterface::class;
                break;
        }

        foreach ($this->processors as $processor) {
            $processor->process($entity);
        }
    }
}
