<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Normalizer;

use DateTime;
use DateTimeImmutable;
use InvalidArgumentException;
use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\CommonBundle\DTO\Id\IdIntegerNullableInterface;
use XCart\Bundle\CommonBundle\DTO\Id\IdUUID4NullableInterface;
use XCart\Bundle\DoctrineBridgeBundle\DTO\IdIntAnnotatedOwnerTrait;
use XCart\Bundle\DoctrineBridgeBundle\DTO\IdUUID4OwnerTrait;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineField;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DoctrineEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\Service\Normalizer\EntityProcessor\EntityProcessorInterface;

/**
 * @property iterable<int,EntityProcessorInterface> $processors
 */
class DoctrineEntityNormalizer implements NormalizerInterface
{
    public function __construct(
        private DataProvider $dataProvider,
        private iterable $processors,
    ) {
    }

    public function normalize(): void
    {
        foreach ($this->dataProvider as $entity) {
            $this->normalizeEntity($entity);
        }
    }

    private function normalizeEntity(Entity $entity): void
    {
        $entity->uses[] = DateTime::class;
        $entity->uses[] = DateTimeImmutable::class;

        switch ($entity->idType) {
            case $entity::ID_TYPE_INT:
                $entity->interfaces[] = IdIntegerNullableInterface::class;
                $entity->traits[] = IdIntAnnotatedOwnerTrait::class;
                break;

            case $entity::ID_TYPE_UUID4:
                $entity->interfaces[] = IdUUID4NullableInterface::class;
                $entity->traits[] = IdUUID4OwnerTrait::class;
                break;
        }

        $ids = [];
        foreach ($entity->indexes as $i => $index) {
            if (in_array($index->name, $ids, true)) {
                unset($entity->indexes[$i]);
                continue;
            }

            $ids[] = $index->name;

            if (empty($index->fields)) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Doctrine entity "%s" has index "%s" with empty fields list',
                        $entity->className,
                        $index->name,
                    )
                );
            }

            $fields = $index->fields;
            foreach ($entity->fields as $field) {
                if (in_array($field->name, $fields)) {
                    unset($fields[array_search($field->name, $fields)]);
                }
            }

            if ($fields) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Doctrine entity "%s" has index "%s" with unknown fields (%s)',
                        $entity->className,
                        $index->name,
                        implode(', ', $fields),
                    )
                );
            }

            $entity->annotations->addAnnotationFromArray(
                ['name' => $index->name, 'columns' => $index->fields],
                '@ORM\Index',
            );
        }

        $ids = [];
        foreach ($entity->uniqueIndexes as $i => $index) {
            if (in_array($index->name, $ids, true)) {
                unset($entity->indexes[$i]);
                continue;
            }

            $ids[] = $index->name;

            if (empty($index->fields)) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Doctrine entity "%s" has unique index "%s" with empty fields list',
                        $entity->className,
                        $index->name,
                    )
                );
            }

            $fields = $index->fields;
            foreach ($entity->fields as $field) {
                if (in_array($field->name, $fields)) {
                    unset($fields[array_search($field->name, $fields)]);
                }
            }

            if ($fields) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Doctrine entity "%s" has unique index "%s" with unknown fields (%s)',
                        $entity->className,
                        $index->name,
                        implode(', ', $fields),
                    )
                );
            }

            $entity->annotations->addAnnotationFromArray(
                ['name' => $index->name, 'columns' => $index->fields],
                '@ORM\UniqueConstraint',
            );
        }

        $has_id = false;

        foreach ($entity->fields as $field) {
            if ($field->id) {
                if ($has_id) {
                    throw new InvalidArgumentException(
                        sprintf('Doctrine entity "%s" has some id field ("%s" field is duplicate)', $entity->className, $field->name)
                    );
                }

                if (!$field->annotations->hasAnnotation('ORM\Id')) {
                    $field->annotations->addAnnotationFromArray([], 'ORM\Id');
                }

                if (!$field->annotations->hasAnnotation('ORM\GeneratedValue')) {
                    $field->annotations->addAnnotationFromArray(['strategy' => 'AUTO'], 'ORM\GeneratedValue');
                }

                $has_id = true;
            }
            $field->annotations->addAnnotationFromArray($this->prepareORMColumnArguments($field), 'ORM\Column');
        }

        foreach ($entity->factory->interfaces as $i => $interface) {
            $entity->factory->uses[] = $interface;
            $entity->factory->interfaces[$i] = ClassNameAssembler::assembleShortName($interface);
        }

        foreach ($this->processors as $processor) {
            $processor->process($entity);
        }
    }

    private function prepareORMColumnArguments(DoctrineField $field): array
    {
        $arguments = ['type' => $field->type, 'options' => []];

        if ($field->nullable) {
            $arguments['nullable'] = true;
        }

        switch ($field->type) {
            case 'string':
                if (!empty($field->length)) {
                    $arguments['length'] = $field->length;
                }
                if (!empty($field->fixed)) {
                    $arguments['options']['fixed'] = true;
                }
                break;

            case 'float':
                if (!empty($field->precision)) {
                    $arguments['precision'] = $field->precision;
                }
                if (!empty($field->scale)) {
                    $arguments['scale'] = $field->scale;
                }
                break;

            case 'DateTimeImmutable':
            case 'DateTime':
                $arguments['type'] = 'datetime';
                break;
        }

        return $arguments;
    }
}
