<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Normalizer;

use DateTime;
use DateTimeImmutable;
use Symfony\Component\Serializer\Annotation\Groups;
use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\DoctrineBridgeBundle\DTO\Filter\FilterInterface as DoctrineFilterInterface;
use XCart\Bundle\DoctrineBridgeBundle\DTO\OrderRule\OrderRuleInterface as DoctrineOrderRuleInterface;
use XCart\Bundle\DoctrineBridgeBundle\Enum\OrderRuleDirection as DoctrineOrderRuleDirection;
use XCart\Bundle\DoctrineBridgeBundle\Factory\FilterFactoryInterface as DoctrineFilterFactoryInterface;
use XCart\Bundle\DoctrineBridgeBundle\Factory\OrderRuleFactoryInterface as DoctrineOrderRuleFactoryInterface;
use XCart\Bundle\LogicBundle\DTO\Request\Filter\FilterInterface;
use XCart\Bundle\LogicBundle\DTO\Request\OrderRule\OrderRuleInterface;
use XCart\Bundle\LogicBundle\Enum\OrderRuleDirection;
use XCart\Bundle\LogicBundle\Factory\Request\FilterFactoryInterface;
use XCart\Bundle\LogicBundle\Factory\Request\OrderRuleFactoryInterface;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DTO as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTODataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\DTOType;
use XCart\Bundle\DTOGeneratorBundle\Service\Normalizer\EntityProcessor\EntityProcessorInterface;

/**
 * @property iterable<int,EntityProcessorInterface> $processors
 */
class DTONormalizer implements NormalizerInterface
{
    public function __construct(
        private DataProvider $dataProvider,
        private iterable $processors,
    ) {
    }

    public function normalize(): void
    {
        foreach ($this->dataProvider as $entity) {
            $this->normalizeEntity($entity);
        }
    }

    private function normalizeEntity(Entity $entity): void
    {
        $entity->uses[] = DateTime::class;
        $entity->uses[] = DateTimeImmutable::class;
        $entity->uses[] = Groups::class;

        foreach ($this->processors as $processor) {
            $processor->process($entity);
        }

        switch ($entity->type?->getValue()) {
            case DTOType::LOGIC_REQUEST_GET_LIST_FILTER:
                $entity->interfaces[] = FilterInterface::class;
                $entity->factory->interfaces[] = FilterFactoryInterface::class;
                $entity->collection = false;

                foreach ($entity->fields as $field) {
                    if (!ClassNameAssembler::isFullClassName($field->type)) {
                        $field->type = 'XCart\Bundle\LogicBundle\DTO\Request\Filter\Value\\' . $field->type;
                    }
                }
                break;

            case DTOType::LOGIC_REQUEST_GET_LIST_ORDER_RULE:
                $entity->interfaces[] = OrderRuleInterface::class;
                $entity->factory->interfaces[] = OrderRuleFactoryInterface::class;
                $entity->collection = false;

                foreach ($entity->fields as $field) {
                    $field->type = OrderRuleDirection::class;
                    $field->nullable = true;
                }
                break;

            case DTOType::DOCTRINE_ENTITY_FILTER:
                $entity->interfaces[] = DoctrineFilterInterface::class;
                $entity->factory->interfaces[] = DoctrineFilterFactoryInterface::class;
                $entity->collection = false;

                foreach ($entity->fields as $field) {
                    if (!ClassNameAssembler::isFullClassName($field->type)) {
                        $field->type = 'XCart\Bundle\DoctrineBridgeBundle\DTO\Filter\Value\\' . $field->type;
                    }
                }
                break;

            case DTOType::DOCTRINE_ENTITY_ORDER_RULE:
                $entity->interfaces[] = DoctrineOrderRuleInterface::class;
                $entity->factory->interfaces[] = DoctrineOrderRuleFactoryInterface::class;
                $entity->collection = false;

                foreach ($entity->fields as $field) {
                    $field->type = DoctrineOrderRuleDirection::class;
                    $field->nullable = true;
                }
                break;
        }

        foreach ($entity->fields as $field) {
            if ($field->groups) {
                $field->annotations->addAnnotationFromArray(['groups' => $field->groups], 'Groups');
            }
        }
    }
}
