<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Normalizer;

use ApiPlatform\Core\Annotation\ApiFilter;
use ApiPlatform\Core\Annotation\ApiProperty;
use ApiPlatform\Core\Annotation\ApiResource;
use DateTime;
use DateTimeImmutable;
use Symfony\Component\Serializer\Annotation\Groups;
use XCart\Bundle\APIPlatformBridgeBundle\API\Entity\EntityInterface;
use XCart\Bundle\APIPlatformBridgeBundle\DTO\IdIntOwnerTrait;
use XCart\Bundle\APIPlatformBridgeBundle\DTO\IdUUID4OwnerTrait;
use XCart\Bundle\CommonBundle\DTO\Id\IdIntegerNullableInterface;
use XCart\Bundle\CommonBundle\DTO\Id\IdUUID4NullableInterface;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Annotation;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\APIEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\APIEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\Service\Normalizer\EntityProcessor\EntityProcessorInterface;

/**
 * @property iterable<int,EntityProcessorInterface> $processors
 */
class APIEntityNormalizer implements NormalizerInterface
{
    public function __construct(
        private DataProvider $dataProvider,
        private iterable     $processors,
    )
    {
    }

    public function normalize(): void
    {
        foreach ($this->dataProvider as $entity) {
            $this->normalizeEntity($entity);
        }
    }

    private function normalizeEntity(Entity $entity): void
    {
        $entity->uses[] = ApiResource::class;
        $entity->uses[] = ApiFilter::class;
        $entity->uses[] = ApiProperty::class;
        $entity->uses[] = DateTime::class;
        $entity->uses[] = DateTimeImmutable::class;
        $entity->uses[] = Groups::class;
        $entity->interfaces[] = EntityInterface::class;

        $entity->path = trim($entity->path, '/');
        if (!$entity->annotations->hasAnnotation('ApiResource')) {
            $entity->annotations->addAnnotationFromArray([], 'ApiResource');
        }

        $idPattern = $entity->idType === $entity::ID_TYPE_UUID4
            ? '[a-zA-Z0-9\-]{36}'
            : '\d+';

        $idType = $entity->idType === $entity::ID_TYPE_UUID4
            ? 'string'
            : 'integer';

        $data = $entity->annotations->getAnnotations('ApiResource')[0]->arguments;

        $data['shortName'] = $entity->name;

        $data['paginationPartial'] = true;

        $data['normalizationContext'] = $entity->normalizationContext
            ? ['groups' => $entity->normalizationContext->groups]
            : ['groups' => ['read', 'insert', 'update']];

        $data['denormalizationContext'] = $entity->denormalizationContext
            ? ['groups' => $entity->denormalizationContext->groups]
            : ['groups' => ['read', 'insert', 'update']];

        $data['itemOperations'] = $data['itemOperations'] ?? [];
        foreach ($entity->itemOperations as $operationName => $operation) {
            $data['itemOperations'][$operationName] = [
                'method' => $operation->method,
                'path'   => $operation->path,
            ];

            if ($operation->identifiers) {
                $data['itemOperations'][$operationName]['identifiers'] = $operation->identifiers;
            }

            if ($operation->requirements) {
                $data['itemOperations'][$operationName]['requirements'] = $operation->requirements;
            }

            if ($operation->openapiContext) {
                $data['itemOperations'][$operationName]['openapi_context'] = $operation->openapiContext;
            }

            if ($operation->normalizationContext) {
                $data['itemOperations'][$operationName]['normalization_context'] = [
                    'groups' => $operation->normalizationContext->groups,
                ];
            }

            if ($operation->denormalizationContext) {
                $data['itemOperations'][$operationName]['denormalization_context'] = [
                    'groups' => $operation->denormalizationContext->groups,
                ];
            }
        }

        $data['collectionOperations'] = $data['collectionOperations'] ?? [];
        foreach ($entity->collectionOperations as $operationName => $operation) {
            $data['collectionOperations'][$operationName] = [
                'method' => $operation->method,
                'path'   => $operation->path,
            ];

            if ($operation->identifiers) {
                $data['collectionOperations'][$operationName]['identifiers'] = $operation->identifiers;
            }

            if ($operation->requirements) {
                $data['collectionOperations'][$operationName]['requirements'] = $operation->requirements;
            }

            if ($operation->openapiContext) {
                $data['collectionOperations'][$operationName]['openapi_context'] = $operation->openapiContext;
            }

            if ($operation->normalizationContext) {
                $data['collectionOperations'][$operationName]['normalization_context'] = [
                    'groups' => $operation->normalizationContext->groups,
                ];
            }

            if ($operation->denormalizationContext) {
                $data['collectionOperations'][$operationName]['denormalization_context'] = [
                    'groups' => $operation->denormalizationContext->groups,
                ];
            }
        }

        $data['attributes'] = ['pagination_client_items_per_page' => true];

        $entityName = self::prepareEntityName($entity->name);

        if ($entity->registerGetOne && !isset($data['itemOperations']['get'])) {
            $data['itemOperations'] = [
                    'get' => [
                        'method'                => 'GET',
                        'path'                  => sprintf('/%s/{id}.{_format}', $entity->path),
                        'identifiers'           => ['id'],
                        'requirements'          => ['id' => $idPattern],
                        'normalization_context' => ['groups' => ['read']],
                        'openapi_context'       => [
                            'summary'    => "Retrieve {$entityName} by id",
                            'parameters' => [
                                [
                                    'name'     => 'id',
                                    'in'       => 'path',
                                    'required' => true,
                                    'schema'   => [
                                        'type' => $idType,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ] + $data['itemOperations'];
        }

        if ($entity->registerFullUpdateOne && !isset($data['itemOperations']['put'])) {
            $data['itemOperations'] = [
                    'put' => [
                        'method'                  => 'PUT',
                        'path'                    => sprintf('/%s/{id}.{_format}', $entity->path),
                        'identifiers'             => ['id'],
                        'requirements'            => ['id' => $idPattern],
                        'normalization_context'   => ['groups' => ['read']],
                        'denormalization_context' => ['groups' => ['update']],
                        'openapi_context'         => [
                            'summary'    => "Change {$entityName} by id",
                            'parameters' => [
                                [
                                    'name'     => 'id',
                                    'in'       => 'path',
                                    'required' => true,
                                    'schema'   => [
                                        'type' => $idType,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ] + $data['itemOperations'];
        }

        if ($entity->registerUpdateOne && !isset($data['itemOperations']['patch'])) {
            $data['itemOperations'] = [
                    'patch' => [
                        'method'                  => 'PATCH',
                        'path'                    => sprintf('/%s/{id}.{_format}', $entity->path),
                        'identifiers'             => ['id'],
                        'requirements'            => ['id' => $idPattern],
                        'normalization_context'   => ['groups' => ['read']],
                        'denormalization_context' => ['groups' => ['update']],
                        'openapi_context'         => [
                            'summary'    => "Partially change {$entityName} by id",
                            'parameters' => [
                                [
                                    'name'     => 'id',
                                    'in'       => 'path',
                                    'required' => true,
                                    'schema'   => [
                                        'type' => $idType,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ] + $data['itemOperations'];
        }

        if ($entity->registerDeleteOne && !isset($data['itemOperations']['delete'])) {
            $data['itemOperations'] = [
                    'delete' => [
                        'method'          => 'DELETE',
                        'path'            => sprintf('/%s/{id}.{_format}', $entity->path),
                        'identifiers'     => ['id'],
                        'requirements'    => ['id' => $idPattern],
                        'openapi_context' => [
                            'summary'    => "Delete {$entityName} by id",
                            'parameters' => [
                                [
                                    'name'     => 'id',
                                    'in'       => 'path',
                                    'required' => true,
                                    'schema'   => [
                                        'type' => $idType,
                                    ],
                                ],
                            ],
                        ],
                    ],
                ] + $data['itemOperations'];
        }

        if ($entity->registerGetCollection && !isset($data['collectionOperations']['get'])) {
            $data['collectionOperations'] = [
                    'get' => [
                        'method'                => 'GET',
                        'path'                  => sprintf('/%s.{_format}', $entity->path),
                        'normalization_context' => ['groups' => ['read']],
                        'openapi_context'       => [
                            'summary' => "Retrieve {$entityName} list",
                        ],
                    ],
                ] + $data['collectionOperations'];
        }

        if ($entity->registerCreateOne && !isset($data['collectionOperations']['post'])) {
            $data['collectionOperations'] = [
                    'post' => [
                        'method'                  => 'POST',
                        'path'                    => sprintf('/%s.{_format}', $entity->path),
                        'denormalization_context' => ['groups' => ['insert']],
                        'openapi_context'         => [
                            'summary' => "Create {$entityName}",
                        ],
                    ],
                ] + $data['collectionOperations'];
        }

        $entity->annotations->setAnnotation('ApiResource', $data);

        foreach ($entity->filters as $filter) {
            $annotation = new Annotation();
            $annotation->name = 'ApiFilter';
            $annotation->arguments = [
                'filterClass' => $filter->className,
                'properties'  => $filter->properties,
                'arguments'   => ['operations' => $filter->operations] + $filter->arguments,
            ];

            $entity->annotations[] = $annotation;
        }

        switch ($entity->idType) {
            case $entity::ID_TYPE_INT:
                $entity->interfaces[] = IdIntegerNullableInterface::class;
                $entity->traits[] = IdIntOwnerTrait::class;
                break;

            case $entity::ID_TYPE_UUID4:
                $entity->interfaces[] = IdUUID4NullableInterface::class;
                $entity->traits[] = IdUUID4OwnerTrait::class;
                break;
        }

        foreach ($entity->fields as $field) {
            if ($field->groups) {
                $field->annotations->addAnnotationFromArray(['groups' => $field->groups], 'Groups');
            }
        }

        foreach ($this->processors as $processor) {
            $processor->process($entity);
        }
    }

    private static function prepareEntityName(string $string): string
    {
        if ($string === '') {
            return 'an entity';
        }

        $string = strtolower(trim($string));
        $string = str_replace(['detailed', 'compact'], '', $string);

        if (str_starts_with($string, 'user ')) {
            return "a {$string}";
        }

        return in_array($string[0], ['a', 'e', 'i', 'o', 'u', 'y'], true)
            ? "an {$string}"
            : "a {$string}";
    }
}
