<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service;

use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DTO;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Transformer;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicResponseType;

class NameResolver implements NameResolverInterface
{
    public function __construct(
        private string $namespace,
        private string $appPrefix,
    ) {
    }

    public function buildCollectionClassName(string $className): string
    {
        return $className . 'Collection';
    }

    public function buildFactoryClassName(string $className): string
    {
        return $this->namespace . '\\Factory\\' . substr($className, strlen($this->namespace) + 1);
    }

    public function buildFactoryInterfaceName(string $className): string
    {
        return $this->buildFactoryClassName($className) . 'Interface';
    }

    public function buildTransformerClassName(Transformer $entity): string
    {
        $fromNamespace = str_starts_with($entity->from, $this->namespace)
            ? substr($entity->from, strlen($this->namespace) + 1)
            : $entity->from;

        $toNamespace = str_starts_with($entity->to, $this->namespace)
            ? substr($entity->to, strlen($this->namespace) + 1)
            : $entity->to;

        return $this->namespace . '\\Transformer\\'
            . $fromNamespace . '\\'
            . $toNamespace;
    }

    public function buildTransformerInterfaceName(Transformer $entity): string
    {
        return $this->buildTransformerClassName($entity) . 'Interface';
    }

    public function buildNewEntityClassName(string $className): string
    {
        return ClassNameAssembler::assembleNamespace($className)
            . '\New' . ClassNameAssembler::assembleShortName($className);
    }

    public function buildLogicRequestClassName(string $className, LogicRequestType $type): string
    {
        $label = match ($type->getValue()) {
            LogicRequestType::GET_ONE           => 'GetOne',
            LogicRequestType::GET_COLLECTION    => 'GetList',
            LogicRequestType::CREATE_ONE        => 'CreateOne',
            LogicRequestType::CREATE_COLLECTION => 'CreateList',
            LogicRequestType::UPDATE_ONE        => 'UpdateOne',
            LogicRequestType::UPDATE_COLLECTION => 'UpdateList',
            LogicRequestType::DELETE_ONE        => 'DeleteOne',
            LogicRequestType::DELETE_COLLECTION => 'DeleteList',
        };

        return preg_replace('/\\\\Entity\\\\/S', '\Action\\', $className) . '\\' . $label . '\DTO\Request';
    }

    public function buildLogicResponseClassName(string $className, LogicResponseType $type): string
    {
        $label = match ($type->getValue()) {
            LogicResponseType::GET_ONE           => 'GetOne',
            LogicResponseType::GET_COLLECTION    => 'GetList',
            LogicResponseType::CREATE_ONE        => 'CreateOne',
            LogicResponseType::CREATE_COLLECTION => 'CreateList',
            LogicResponseType::UPDATE_ONE        => 'UpdateOne',
            LogicResponseType::UPDATE_COLLECTION => 'UpdateList',
            LogicResponseType::DELETE_ONE        => 'DeleteOne',
            LogicResponseType::DELETE_COLLECTION => 'DeleteList',
        };

        return preg_replace('/\\\\Entity\\\\/S', '\Action\\', $className) . '\\' . $label . '\DTO\Response';
    }

    public function buildRepositoryClassName(string $className): string
    {
        return str_replace('\Entity\\', '\Repository\\', $className);
    }

    public function buildRepositoryReadId(string $className): string
    {
        return sprintf(
            '%s.repository.%s.read',
            $this->appPrefix,
            $this->buildDoctrineShortServiceSuffix($className),
        );
    }

    public function buildRepositoryWriteId(string $className): string
    {
        return sprintf(
            '%s.repository.%s.write',
            $this->appPrefix,
            $this->buildDoctrineShortServiceSuffix($className),
        );
    }

    public function buildDataSourceReadClassName(string $className): string
    {
        return str_replace('\Entity\\', '\DataSource\\', $className) . 'Read';
    }

    public function buildDataSourceWriteClassName(string $className): string
    {
        return str_replace('\Entity\\', '\DataSource\\', $className) . 'Write';
    }

    public function buildDataSourceReadId(string $className): string
    {
        return sprintf(
            '%s.data_source.%s.read',
            $this->appPrefix,
            $this->buildDoctrineShortServiceSuffix($className),
        );
    }

    public function buildDataSourceWriteId(string $className): string
    {
        return sprintf(
            '%s.data_source.%s.write',
            $this->appPrefix,
            $this->buildDoctrineShortServiceSuffix($className),
        );
    }

    public function buildDataSourceReadInterfaceName(string $className): string
    {
        return $this->buildDataSourceReadClassName($className) . 'Interface';
    }

    public function buildDataSourceWriteInterfaceName(string $className): string
    {
        return $this->buildDataSourceWriteClassName($className) . 'Interface';
    }

    public function buildDoctrineGetListCriteriaAssemblerId(LogicEntity $entity): string
    {
        return sprintf(
            '%s.doctrine.%s.criteria.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($entity->className),
        );
    }

    public function buildDoctrineFilterTransformerId(LogicEntity|DTO $entity): string
    {
        if ($entity instanceof LogicEntity) {
            return sprintf(
                '%s.doctrine.%s.filter.transformer',
                $this->appPrefix,
                $this->buildLogicShortServiceSuffix($entity->className),
            );
        }

        $suffix = preg_replace('/^.+\\\\DTO\\\\Doctrine\\\\/Ss', '', $entity->className);
        $suffix = str_replace('\Filter', '', $suffix);
        $suffix = strtolower(str_replace('\\', '.', $suffix));

        return sprintf(
            '%s.doctrine.%s.filter.transformer',
            $this->appPrefix,
            $suffix,
        );
    }

    public function buildDoctrineOrderRuleTransformerId(LogicEntity|DTO $entity): string
    {
        if ($entity instanceof LogicEntity) {
            return sprintf(
                '%s.doctrine.%s.order_rule.transformer',
                $this->appPrefix,
                $this->buildLogicShortServiceSuffix($entity->className),
            );
        }

        $suffix = preg_replace('/^.+\\\\DTO\\\\Doctrine\\\\/Ss', '', $entity->className);
        $suffix = str_replace('\OrderRule', '', $suffix);
        $suffix = strtolower(str_replace('\\', '.', $suffix));

        return sprintf(
            '%s.doctrine.%s.order_rule.transformer',
            $this->appPrefix,
            $suffix,
        );
    }

    public function buildQueryBuilderClassName(string $className): string
    {
        return str_replace('\Entity\\', '\QueryBuilder\\', $className);
    }

    public function buildQueryBuilderFactoryReadID(string $className): string
    {
        return sprintf(
            '%s.query_builder.%s.read',
            $this->appPrefix,
            $this->buildDoctrineShortServiceSuffix($className),
        );
    }

    public function buildQueryBuilderFactoryWriteID(string $className): string
    {
        return sprintf(
            '%s.query_builder.%s.write',
            $this->appPrefix,
            $this->buildDoctrineShortServiceSuffix($className),
        );
    }

    public function buildQueryBuilderInterfaceName(string $className): string
    {
        return $this->buildQueryBuilderClassName($className) . 'Interface';
    }

    public function buildQueryBuilderReadInterfaceName(string $className): string
    {
        return $this->buildQueryBuilderClassName($className) . 'ReadInterface';
    }

    public function buildQueryBuilderWriteInterfaceName(string $className): string
    {
        return $this->buildQueryBuilderClassName($className) . 'WriteInterface';
    }

    public function buildRepositoryReadInterfaceName(string $className): string
    {
        return $this->buildRepositoryClassName($className) . 'ReadInterface';
    }

    public function buildRepositoryWriteInterfaceName(string $className): string
    {
        return $this->buildRepositoryClassName($className) . 'WriteInterface';
    }

    public function buildAPIGetOneDataProviderId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.item',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetOneRequestAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.item.request.assembler',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetOneResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.item.response.assembler',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetCollectionDataProviderId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.collection',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetCollectionRequestAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.collection.request.assembler',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetCollectionRequestPaginationEnricherId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.collection.request.enricher.pagintation',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetCollectionRequestFilterAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.collection.request.assembler.filter',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetCollectionRequestOrderRuleAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.collection.request.assembler.order_rule',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIGetCollectionResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_provider.collection.response.assembler',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIDataPersisterId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIPersisterId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.persister',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIPersisterCreateRequestAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.persister.request.assembler.create',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIPersisterUpdateRequestAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.persister.request.assembler.update',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIPersisterCreateResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.persister.response.assembler.create',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIPersisterUpdateResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.persister.response.assembler.update',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIRemoverId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.remover',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildAPIRemoverRequestAssemblerId(string $className): string
    {
        return sprintf(
            '%s.api.%s.data_persister.remover.request.assembler',
            $this->appPrefix,
            $this->buildAPIShortServiceSuffix($className),
        );
    }

    public function buildLogicGetOneId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.get_one.action',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicGetOneEntityIdAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.get_one.action.entity_id.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicGetCollectionId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.get_collection.action',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicCreateOneId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.create_one.action',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicUpdateOneId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.update_one.action',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicDeleteOneId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.delete_one.action',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicEntityFilterClassName(string $className): string
    {
        return $this->buildLogicRequestClassName($className, LogicRequestType::GET_COLLECTION()) . '\Filter';
    }

    public function buildLogicEntityOrderRuleClassName(string $className): string
    {
        return $this->buildLogicRequestClassName($className, LogicRequestType::GET_COLLECTION()) . '\OrderRule';
    }

    public function buildLogicGetOneResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.get_one.response.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicGetCollectionResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.get_collection.response.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicCreateOneEntityAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.create_one.entity.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicCreateOneResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.create_one.response.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicUpdateOneEntityEnricherId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.update_one.entity.enricher',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicUpdateOneEntityInternalEnricherId(string $className): string
    {
        return $this->buildLogicUpdateOneEntityEnricherId($className) . '.internal';
    }

    public function buildLogicUpdateOneResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.update_one.response.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicDeleteOneArgumentsAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.delete_one.arguments.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildLogicDeleteOneResponseAssemblerId(string $className): string
    {
        return sprintf(
            '%s.logic.%s.delete_one.response.assembler',
            $this->appPrefix,
            $this->buildLogicShortServiceSuffix($className),
        );
    }

    public function buildDoctrineGetListFilterClassName(LogicEntity $entity): string
    {
        return str_replace('\Logic\Entity\\', '\DTO\Doctrine\\', $entity->className) . '\Filter';
    }

    public function buildDoctrineGetListOrderRuleClassName(LogicEntity $entity): string
    {
        return str_replace('\Logic\Entity\\', '\DTO\Doctrine\\', $entity->className) . '\OrderRule';
    }

    private function buildAPIShortServiceSuffix(string $className): string
    {
        return strtolower(str_replace('\\', '.', preg_replace('/^.+\\\\Entity\\\\/Ss', '', $className)));
    }

    private function buildLogicShortServiceSuffix(string $className): string
    {
        return strtolower(str_replace('\\', '.', preg_replace('/^.+\\\\Entity\\\\/Ss', '', $className)));
    }

    private function buildDoctrineShortServiceSuffix(string $className): string
    {
        if (str_contains($className, '\Entity\\')) {
            return strtolower(str_replace('\\', '.', preg_replace('/^.+\\\\Entity\\\\/Ss', '', $className)));
        }

        return strtolower(str_replace('\\', '.', preg_replace('/^.+\\\\Model\\\\/Ss', '', $className)));
    }
}
