<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service;

use InvalidArgumentException;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;
use XCart\Bundle\CommonBundle\Enricher\Transparent\WalkingTransparentEnricher as UpdateOneEntityInternalEnricher;
use XCart\Bundle\DoctrineBridgeBundle\Action\CreateOneAction;
use XCart\Bundle\DoctrineBridgeBundle\Action\DeleteOneAction;
use XCart\Bundle\DoctrineBridgeBundle\Action\GetListAction;
use XCart\Bundle\DoctrineBridgeBundle\Action\GetOneAction;
use XCart\Bundle\DoctrineBridgeBundle\Action\UpdateOneAction;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\CreateOne\Entity\TransformerWrapperAssembler as CreateOneEntityAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\DeleteOne\Arguments\ArgumentsAssemblerInterface as DeleteOneArgumentsAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\DeleteOne\Arguments\IdBasedArgumentsAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\DeleteOne\Response\EmptyWrapperAssembler as DeleteOneResponseAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetList\Criteria\FindListCriteriaAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetList\Response\TransformerWrapperAssembler as GetListResponseAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetOne\EntityId\EntityIdAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetOne\Response\TransformerWrapperAssembler as GetOneResponseTransformer;
use XCart\Bundle\DoctrineBridgeBundle\Enricher\UpdateOne\TransparentEnricher as UpdateOneEntityEnricher;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Transformer;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\TransformersDataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicResponseType;
use XCart\Bundle\LogicBundle\DependencyInjection\LogicExtension;

class LogicCRUDRegister
{
    public function __construct(
        private NameResolverInterface $nameResolver,
        private LogicEntitiesDataProvider $dataProvider,
        private TransformersDataProvider $transformersDataProvider,
    ) {
    }

    public function register(ContainerBuilder $container): void
    {
        /** @var LogicEntity $entity */
        foreach ($this->dataProvider as $entity) {
            if ($entity->useDoctrineGetOne) {
                $this->registerGetOne($container, $entity);
            }

            if ($entity->useDoctrineGetCollection) {
                $this->registerGetCollection($container, $entity);
            }

            if ($entity->useDoctrineCreateOne) {
                $this->registerCreateOne($container, $entity);
            }

            if ($entity->useDoctrineUpdateOne) {
                $this->registerUpdateOne($container, $entity);
            }

            if ($entity->useDoctrineDeleteOne) {
                $this->registerDeleteOne($container, $entity);
            }
        }
    }

    private function registerGetOne(ContainerBuilder $container, LogicEntity $entity): void
    {
        if (!$entity->doctrineEntity) {
            throw new InvalidArgumentException(
                sprintf('Doctrine entity not declared for logic entity "%s"', $entity->className)
            );
        }

        $requestClassName = $this->nameResolver
            ->buildLogicRequestClassName($entity->className, LogicRequestType::GET_ONE());
        $responseClassName = $this->nameResolver
            ->buildLogicResponseClassName($entity->className, LogicResponseType::GET_ONE());

        // action
        if (!$container->hasDefinition($this->nameResolver->buildLogicGetOneId($entity->className))) {
            $definition = (new Definition(GetOneAction::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->addTag(
                    LogicExtension::ACTION_TAG,
                    [
                        'request'  => $requestClassName,
                        'response' => $responseClassName,
                    ],
                )
                ->setArgument('$dataSource', new Reference($this->nameResolver->buildDataSourceReadId($entity->doctrineEntity)))
                ->setArgument('$entityIdAssembler', new Reference($this->nameResolver->buildLogicGetOneEntityIdAssemblerId($entity->className)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildLogicGetOneResponseAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildLogicGetOneId($entity->className), $definition);
        }

        // entity id assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicGetOneEntityIdAssemblerId($entity->className))) {
            $definition = (new Definition(EntityIdAssembler::class));

            $container->setDefinition($this->nameResolver->buildLogicGetOneEntityIdAssemblerId($entity->className), $definition);
        }

        // response assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicGetOneResponseAssemblerId($entity->className))) {
            $transformer = $this->findTransformer($entity->doctrineEntity, $responseClassName);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from doctrine entity "%s" to logic response "%s"',
                        $entity->doctrineEntity,
                        $responseClassName,
                    )
                );
            }

            $definition = (new Definition(GetOneResponseTransformer::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($this->nameResolver->buildLogicGetOneResponseAssemblerId($entity->className), $definition);
        }
    }

    private function registerGetCollection(ContainerBuilder $container, LogicEntity $entity): void
    {
        if (!$entity->doctrineEntity) {
            throw new InvalidArgumentException(
                sprintf('Doctrine entity not declared for logic entity "%s"', $entity->className)
            );
        }

        $requestClassName = $this->nameResolver
            ->buildLogicRequestClassName($entity->className, LogicRequestType::GET_COLLECTION());
        $responseClassName = $this->nameResolver
            ->buildLogicResponseClassName($entity->className, LogicResponseType::GET_COLLECTION());

        // action
        if (!$container->hasDefinition($this->nameResolver->buildLogicGetCollectionId($entity->className))) {
            $definition = (new Definition(GetListAction::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->addTag(
                    LogicExtension::ACTION_TAG,
                    [
                        'request'  => $requestClassName,
                        'response' => $responseClassName,
                    ],
                )
                ->setArgument('$readDataSource', new Reference($this->nameResolver->buildDataSourceReadId($entity->doctrineEntity)))
                ->setArgument('$criteriaAssembler', new Reference($this->nameResolver->buildDoctrineGetListCriteriaAssemblerId($entity)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildLogicGetCollectionResponseAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildLogicGetCollectionId($entity->className), $definition);
        }

        // criteria assembler
        if (!$container->hasDefinition($this->nameResolver->buildDoctrineGetListCriteriaAssemblerId($entity))) {
            $definition = (new Definition(FindListCriteriaAssembler::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->setArgument('$filterTransformer', new Reference($this->nameResolver->buildDoctrineFilterTransformerId($entity)))
                ->setArgument('$orderRuleTransformer', new Reference($this->nameResolver->buildDoctrineOrderRuleTransformerId($entity)));

            $container->setDefinition($this->nameResolver->buildDoctrineGetListCriteriaAssemblerId($entity), $definition);
        }

        // response assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicGetCollectionResponseAssemblerId($entity->className))) {
            $transformer = $this->findTransformer($entity->doctrineEntity, $responseClassName);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from doctrine entity "%s" to logic response "%s"',
                        $entity->doctrineEntity,
                        $responseClassName,
                    )
                );
            }

            $definition = (new Definition(GetListResponseAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($this->nameResolver->buildLogicGetCollectionResponseAssemblerId($entity->className), $definition);
        }
    }

    private function registerCreateOne(ContainerBuilder $container, LogicEntity $entity): void
    {
        if (!$entity->doctrineEntity) {
            throw new InvalidArgumentException(
                sprintf('Doctrine entity not declared for logic entity "%s"', $entity->className)
            );
        }

        $requestClassName = $this->nameResolver
            ->buildLogicRequestClassName($entity->className, LogicRequestType::CREATE_ONE());
        $responseClassName = $this->nameResolver
            ->buildLogicResponseClassName($entity->className, LogicResponseType::CREATE_ONE());

        // action
        if (!$container->hasDefinition($this->nameResolver->buildLogicCreateOneId($entity->className))) {
            $definition = (new Definition(CreateOneAction::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->addTag(
                    LogicExtension::ACTION_TAG,
                    [
                        'request'  => $requestClassName,
                        'response' => $responseClassName,
                    ],
                )
                ->setArgument('$writeRepository', new Reference($this->nameResolver->buildRepositoryWriteId($entity->doctrineEntity)))
                ->setArgument('$entityAssembler', new Reference($this->nameResolver->buildLogicCreateOneEntityAssemblerId($entity->className)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildLogicCreateOneResponseAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildLogicCreateOneId($entity->className), $definition);
        }

        // entity assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicCreateOneEntityAssemblerId($entity->className))) {
            $transformer = $this->findTransformer($requestClassName, $entity->doctrineEntity);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic request entity "%s" to doctrine entity "%s"',
                        $requestClassName,
                        $entity->doctrineEntity,
                    )
                );
            }
            $definition = (new Definition(CreateOneEntityAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($this->nameResolver->buildLogicCreateOneEntityAssemblerId($entity->className), $definition);
        }

        // response assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicCreateOneResponseAssemblerId($entity->className))) {
            $transformer = $this->findTransformer($entity->doctrineEntity, $responseClassName);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from doctrine entity "%s" to logic response "%s"',
                        $entity->doctrineEntity,
                        $responseClassName,
                    )
                );
            }
            $definition = (new Definition(GetOneResponseTransformer::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($this->nameResolver->buildLogicCreateOneResponseAssemblerId($entity->className), $definition);
        }
    }

    private function registerUpdateOne(ContainerBuilder $container, LogicEntity $entity): void
    {
        if (!$entity->doctrineEntity) {
            throw new InvalidArgumentException(
                sprintf('Doctrine entity not declared for logic entity "%s"', $entity->className)
            );
        }

        $requestClassName = $this->nameResolver
            ->buildLogicRequestClassName($entity->className, LogicRequestType::UPDATE_ONE());
        $responseClassName = $this->nameResolver
            ->buildLogicResponseClassName($entity->className, LogicResponseType::UPDATE_ONE());

        // action
        if (!$container->hasDefinition($this->nameResolver->buildLogicUpdateOneId($entity->className))) {
            $definition = (new Definition(UpdateOneAction::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->addTag(
                    LogicExtension::ACTION_TAG,
                    [
                        'request'  => $requestClassName,
                        'response' => $responseClassName,
                    ],
                )
                ->setArgument('$writeRepository', new Reference($this->nameResolver->buildRepositoryWriteId($entity->doctrineEntity)))
                ->setArgument('$entityEnricher', new Reference($entity->enricher ?? $this->nameResolver->buildLogicUpdateOneEntityEnricherId($entity->className)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildLogicUpdateOneResponseAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildLogicUpdateOneId($entity->className), $definition);
        }

        if (!$entity->enricher) {
            // entity enricher
            if (!$container->hasDefinition($this->nameResolver->buildLogicUpdateOneEntityEnricherId($entity->className))) {
                $definition = (new Definition(UpdateOneEntityEnricher::class))
                    ->setArgument('$enricher', new Reference($this->nameResolver->buildLogicUpdateOneEntityInternalEnricherId($entity->className)));

                $container->setDefinition($this->nameResolver->buildLogicUpdateOneEntityEnricherId($entity->className), $definition);
            }

            // entity internal enricher
            if (!$container->hasDefinition($this->nameResolver->buildLogicUpdateOneEntityInternalEnricherId($entity->className))) {
                $definition = (new Definition(UpdateOneEntityInternalEnricher::class));

                $container->setDefinition($this->nameResolver->buildLogicUpdateOneEntityInternalEnricherId($entity->className), $definition);
            }
        }

        // response assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicUpdateOneResponseAssemblerId($entity->className))) {
            $transformer = $this->findTransformer($entity->doctrineEntity, $responseClassName);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from doctrine entity "%s" to logic response "%s"',
                        $entity->doctrineEntity,
                        $responseClassName,
                    )
                );
            }
            $definition = (new Definition(GetOneResponseTransformer::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($this->nameResolver->buildLogicUpdateOneResponseAssemblerId($entity->className), $definition);
        }
    }

    private function registerDeleteOne(ContainerBuilder $container, LogicEntity $entity): void
    {
        if (!$entity->doctrineEntity) {
            throw new InvalidArgumentException(
                sprintf('Doctrine entity not declared for logic entity "%s"', $entity->className)
            );
        }

        $requestClassName = $this->nameResolver
            ->buildLogicRequestClassName($entity->className, LogicRequestType::DELETE_ONE());
        $responseClassName = $this->nameResolver
            ->buildLogicResponseClassName($entity->className, LogicResponseType::DELETE_ONE());

        // action
        if (!$container->hasDefinition($this->nameResolver->buildLogicDeleteOneId($entity->className))) {
            $definition = (new Definition(DeleteOneAction::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->addTag(
                    LogicExtension::ACTION_TAG,
                    [
                        'request'  => $requestClassName,
                        'response' => $responseClassName,
                    ],
                )
                ->setArgument('$writeRepository', new Reference($this->nameResolver->buildRepositoryWriteId($entity->doctrineEntity)))
                ->setArgument('$argumentsAssembler', new Reference($this->nameResolver->buildLogicDeleteOneArgumentsAssemblerId($entity->className)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildLogicDeleteOneResponseAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildLogicDeleteOneId($entity->className), $definition);
        }

        // arguments assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicDeleteOneArgumentsAssemblerId($entity->className))) {
            $definition = (new Definition(IdBasedArgumentsAssembler::class))
                ->setArgument('$idFieldName', 'id');

            $container->setDefinition($this->nameResolver->buildLogicDeleteOneArgumentsAssemblerId($entity->className), $definition);
        }

        // response assembler
        if (!$container->hasDefinition($this->nameResolver->buildLogicDeleteOneResponseAssemblerId($entity->className))) {
            $definition = (new Definition(DeleteOneResponseAssembler::class))
                ->setArgument('$factory', new Reference($this->nameResolver->buildFactoryInterfaceName($responseClassName)));

            $container->setDefinition($this->nameResolver->buildLogicDeleteOneResponseAssemblerId($entity->className), $definition);
        }
    }

    private function findTransformer(string $from, string $to): ?Transformer
    {
        /** @var Transformer $transformer */
        foreach ($this->transformersDataProvider as $transformer) {
            if ($transformer->from === $from && $transformer->to === $to) {
                return $transformer;
            }
        }

        return null;
    }
}
