<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Generator;

use Doctrine\Common\Collections\ArrayCollection;
use RuntimeException;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DTOInterface;
use XCart\Bundle\DTOGeneratorBundle\Service\GenerationDataBuilder\GenerationDataBuilderInterface;
use XCart\Bundle\DTOGeneratorBundle\Service\NameResolverInterface;
use XCart\Bundle\DTOGeneratorBundle\Service\RendererInterface;

abstract class AbstractGenerator
{
    abstract protected function getClassName(DTOInterface $entity): string;

    abstract protected function getTemplate(DTOInterface $entity): string;

    public function __construct(
        protected GenerationDataBuilderInterface $dataBuilder,
        protected RendererInterface $render,
        protected NameResolverInterface $nameResolver,
    ) {
    }

    public function generate(string $directory, string $namespace, ArrayCollection $dataProvider): void
    {
        /** @var DTOInterface $entity */
        foreach ($dataProvider as $entity) {
            $this->generateEntity($directory, $namespace, $entity);
        }
    }

    protected function generateEntity(string $directory, string $namespace, DTOInterface $entity): void
    {
        $this->save(
            $this->buildPath($directory, $namespace, $entity),
            $this->render($entity),
        );
    }

    private function render(DTOInterface $entity): string
    {
        return $this->render->render($this->getTemplate($entity), ['entity' => $this->dataBuilder->build($entity)]);
    }

    private function save(string $path, string $content): void
    {
        file_put_contents($path, $content);
    }

    private function buildPath(string $directory, string $namespace, DTOInterface $entity): string
    {
        $classShortNamespace = substr(str_replace($namespace, '', $this->getClassName($entity)), 1);
        $classShortNamespaceParts = explode('\\', $classShortNamespace);

        $path = $directory . '/' . implode('/', $classShortNamespaceParts) . '.php';

        $this->createDirectory(dirname($path));

        return $path;
    }

    private function createDirectory(string $dir): void
    {
        if (!file_exists($dir) && !mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new RuntimeException(sprintf('Directory "%s" was not created', $dir));
        }
    }
}
