<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\GenerationDataBuilder;

use DateTimeImmutable;
use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\CommonBundle\Assembler\MethodNameAssembler;
use XCart\Bundle\CommonBundle\Helper\TypeHelper;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Field;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Transformer;

class TransformerDataBuilder extends AbstractDataBuilder
{
    /**
     * @param Transformer $entity
     */
    public function build($entity): array
    {
        if ($entity->type) {
            if ($entity->type->isLogicResponseItemPayload()) {
                return $this->buildLogicResponseItemPayload($entity);
            }

            if ($entity->type->isLogicResponseCollectionPayload()) {
                return $this->buildLogicResponseCollectionPayload($entity);
            }

            if ($entity->type->isLogicRequestItemPayload()) {
                return $this->buildLogicRequestItemPayload($entity);
            }
        }

        if ($entity->pattern->isDirect()) {
            return $this->buildDirect($entity);
        }

        if ($entity->pattern->isDirectTo()) {
            return $this->buildDirectTo($entity);
        }

        if ($entity->pattern->isDirectFrom()) {
            return $this->buildDirectFrom($entity);
        }

        return $this->buildRegular($entity);
    }

    private function buildLogicResponseItemPayload(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);
        $intermediateTransformerInterfaceName = $this->nameResolver->buildTransformerInterfaceName($entity->intermediateTransformer);

        return [
            'namespace'               => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'          => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort'      => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'           => $entity->from,
            'fromClassNameShort'      => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'             => $entity->to,
            'toClassNameShort'        => ClassNameAssembler::assembleShortName($entity->to),
            'intermediateTransformer' => $intermediateTransformerInterfaceName,
            'toFactoryClassName'      => $entity->toEntity ? $this->nameResolver->buildFactoryInterfaceName($entity->to) : null,
        ];
    }

    private function buildLogicResponseCollectionPayload(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);
        $intermediateTransformerInterfaceName = $this->nameResolver->buildTransformerInterfaceName($entity->intermediateTransformer);

        return [
            'namespace'               => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'          => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort'      => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'           => $entity->from,
            'fromClassNameShort'      => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'             => $entity->to,
            'toClassNameShort'        => ClassNameAssembler::assembleShortName($entity->to),
            'intermediateTransformer' => $intermediateTransformerInterfaceName,
            'toFactoryClassName'      => $entity->toEntity ? $this->nameResolver->buildFactoryInterfaceName($entity->to) : null,
        ];
    }

    private function buildLogicRequestItemPayload(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);
        $intermediateTransformerInterfaceName = $this->nameResolver->buildTransformerInterfaceName($entity->intermediateTransformer);

        return [
            'namespace'               => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'          => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort'      => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'           => $entity->from,
            'fromClassNameShort'      => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'             => $entity->to,
            'toClassNameShort'        => ClassNameAssembler::assembleShortName($entity->to),
            'intermediateTransformer' => $intermediateTransformerInterfaceName,
            'toFactoryClassName'      => $entity->toEntity ? $this->nameResolver->buildFactoryInterfaceName($entity->to) : null,
        ];
    }

    private function buildDirect(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);

        $methods = [];
        /** @var Field $fieldTo */
        foreach ($entity->toEntity->fields as $fieldTo) {
            if (!$this->isAllowedType($fieldTo->type)) {
                continue;
            }

            $found = false;
            /** @var Field $fieldFrom */
            foreach ($entity->fromEntity->fields as $fieldFrom) {
                if ($fieldTo->name === $fieldFrom->name && $this->isAllowedType($fieldFrom->type)) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                continue;
            }

            $methods[] = [
                'getter' => MethodNameAssembler::assemblegetterName($fieldTo->name),
                'setter' => MethodNameAssembler::assembleSetterName($fieldTo->name),
            ];
        }

        return [
            'namespace'          => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'     => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort' => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'      => $entity->from,
            'fromClassNameShort' => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'        => $entity->to,
            'toClassNameShort'   => ClassNameAssembler::assembleShortName($entity->to),
            'toFactoryClassName' => $this->nameResolver->buildFactoryInterfaceName($entity->to),
            'methods'            => $methods,
        ];
    }

    private function buildDirectTo(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);

        $methods = [
            [
                'getter' => 'getId',
                'setter' => 'setId',
            ]
        ];
        /** @var Field $field */
        foreach ($entity->toEntity->fields as $field) {
            if (!$this->isAllowedType($field->type)) {
                continue;
            }

            $methods[] = [
                'getter' => MethodNameAssembler::assemblegetterName($field->name),
                'setter' => MethodNameAssembler::assembleSetterName($field->name),
            ];
        }

        return [
            'namespace'          => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'     => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort' => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'      => $entity->from,
            'fromClassNameShort' => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'        => $entity->to,
            'toClassNameShort'   => ClassNameAssembler::assembleShortName($entity->to),
            'toFactoryClassName' => $this->nameResolver->buildFactoryInterfaceName($entity->to),
            'methods'            => $methods,
        ];
    }

    private function buildDirectFrom(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);

        $methods = [
            [
                'getter' => 'getId',
                'setter' => 'setId',
            ]
        ];
        /** @var Field $field */
        foreach ($entity->fromEntity->fields as $field) {
            if (!$this->isAllowedType($field->type)) {
                continue;
            }

            $methods[] = [
                'getter' => MethodNameAssembler::assemblegetterName($field->name),
                'setter' => MethodNameAssembler::assembleSetterName($field->name),
            ];
        }

        return [
            'namespace'          => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'     => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort' => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'      => $entity->from,
            'fromClassNameShort' => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'        => $entity->to,
            'toClassNameShort'   => ClassNameAssembler::assembleShortName($entity->to),
            'methods'            => $methods,
        ];
    }

    private function buildRegular(Transformer $entity): array
    {
        $className = $this->nameResolver->buildTransformerClassName($entity);
        $interfaceName = $this->nameResolver->buildTransformerInterfaceName($entity);

        return [
            'namespace'          => ClassNameAssembler::assembleNamespace($className),
            'classNameShort'     => ClassNameAssembler::assembleShortName($className),
            'interfaceNameShort' => ClassNameAssembler::assembleShortName($interfaceName),
            'fromClassName'      => $entity->from,
            'fromClassNameShort' => ClassNameAssembler::assembleShortName($entity->from),
            'toClassName'        => $entity->to,
            'toClassNameShort'   => ClassNameAssembler::assembleShortName($entity->to),
        ];
    }

    private function isAllowedType(string $type): bool
    {
        return !TypeHelper::isClassType($type)
            || in_array($type, [DateTimeImmutable::class], true);
    }
}
