<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\GenerationDataBuilder\Entity;

use JsonException;
use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\CommonBundle\Assembler\MethodNameAssembler;
use XCart\Bundle\DTOGeneratorBundle\Collection\Annotations;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\AbstractEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Annotation;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineField;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\EntityField;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Field;
use XCart\Bundle\DTOGeneratorBundle\DTO\Attribute;
use XCart\Bundle\DTOGeneratorBundle\Service\GenerationDataBuilder\AbstractDataBuilder;

abstract class AbstractEntityDataBuilder extends AbstractDataBuilder
{
    protected bool $useAttributes = true;

    /**
     * @throws JsonException
     */
    protected function buildAttributeObjectArgumentsList(array $arguments): array
    {
        $parts = [];
        foreach ($arguments as $name => $value) {
            $cell = is_int($name)
                ? ''
                : $name . ': ';

            switch (gettype($value)) {
                case 'integer':
                case 'double':
                    $cell .= $value;
                    break;

                case 'boolean':
                    $cell .= $value ? 'true' : 'false';
                    break;

                case 'array':
                    $cell .= '[' . implode(', ', $this->buildArray($value)) . ']';
                    break;

                case 'object':
                    if ($value instanceof Attribute) {
                        $cell .= 'new ' . (ClassNameAssembler::isFullClassName($value->getClassName()) ? '\\' : '') . $value->getClassName() . '('
                            . implode(', ', $this->buildAttributeObjectArgumentsList($value->getArguments()))
                            . ')';
                    } elseif ($value instanceof Annotation) {
                        $cell .= 'new ' . (ClassNameAssembler::isFullClassName($value->name) ? '\\' : '') . $value->name . '('
                            . implode(', ', $this->buildAttributeObjectArgumentsList($value->arguments))
                            . ')';
                    } else {
                        $cell .= 'new \\' . get_class($value) . '('
                            . implode(', ', $this->buildAttributeObjectArgumentsList(json_decode(json_encode($value, JSON_THROW_ON_ERROR), true, 512, JSON_THROW_ON_ERROR)))
                            . ')';
                    }
                    break;

                case 'string':
                    if (str_ends_with($value, '::class')) {
                        $cell .= $value;
                    } elseif (ClassNameAssembler::isFullClassName($value)) {
                        $cell .= '\\' . trim($value, '\\') . '::class';
                    } else {
                        $cell .= '"' . addcslashes($value, '"') . '"';
                    }
                    break;

                case 'NULL':
                    $cell .= 'null';
                    break;
            }

            $parts[] = $cell;
        }

        return $parts;
    }

    protected function buildAnnotationObjectArgumentsList(array $arguments): array
    {
        $parts = [];
        foreach ($arguments as $name => $value) {
            $cell = is_int($name)
                ? ''
                : $name . '=';

            switch (gettype($value)) {
                case 'integer':
                case 'double':
                    $cell .= $value;
                    break;

                case 'boolean':
                    $cell .= $value ? 'true' : 'false';
                    break;

                case 'array':
                    $cell .= '{' . implode(', ', $this->buildAnnotationArray($value)) . '}';
                    break;

                case 'object':
                    if ($value instanceof Attribute) {
                        $cell .= '@' . $value->getClassName() . '('
                            . implode(', ', $this->buildAnnotationObjectArgumentsList($value->getArguments()))
                            . ')';
                    } elseif ($value instanceof Annotation) {
                        $cell .= '@' .  $value->name . '('
                            . implode(', ', $this->buildAnnotationObjectArgumentsList($value->arguments))
                            . ')';
                    } else {
                        $cell .= '@' . get_class($value) . '('
                            . implode(', ', $this->buildAnnotationObjectArgumentsList(json_decode(json_encode($value, JSON_THROW_ON_ERROR), true, 512, JSON_THROW_ON_ERROR)))
                            . ')';
                    }
                    break;

                case 'string':
                    if (str_ends_with($value, '::class')) {
                        $cell .= '"' . substr($value, 0, -7) . '"';
                    } else {
                        $cell .= '"' . addcslashes($value, '"') . '"';
                    }
                    break;

                case 'NULL':
                    $cell .= 'null';
                    break;
            }

            $parts[] = $cell;
        }

        return $parts;
    }

    /**
     * @throws JsonException
     */
    protected function buildArray(array $arguments): array
    {
        $parts = [];
        $isList = array_is_list($arguments);
        foreach ($arguments as $name => $value) {
            if ($isList && is_array($value) && !empty($value['_'])) {
                $name = $value['_'];
                unset($value['_']);
            }

            $cell = '';

            if (!$isList) {
                $cell = '"' . $name . '" => ';
            }

            switch (gettype($value)) {
                case 'integer':
                case 'double':
                    $cell .= $value;
                    break;

                case 'boolean':
                    $cell .= $value ? 'true' : 'false';
                    break;

                case 'array':
                    $cell .= '[' . implode(', ', $this->buildArray($value)) . ']';
                    break;

                case 'object':
                    if ($value instanceof Attribute) {
                        $cell .= 'new ' . (ClassNameAssembler::isFullClassName($value->getClassName()) ? '\\' : '') . $value->getClassName() . '('
                            . implode(', ', $this->buildAttributeObjectArgumentsList($value->getArguments()))
                            . ')';
                    } elseif ($value instanceof Annotation) {
                        $cell .= 'new ' . (ClassNameAssembler::isFullClassName($value->name) ? '\\' : '') . $value->name . '('
                            . implode(', ', $this->buildAttributeObjectArgumentsList($value->arguments))
                            . ')';
                    } else {
                        $cell .= 'new \\' . get_class($value) . '('
                            . implode(', ', $this->buildAttributeObjectArgumentsList(json_decode(json_encode($value, JSON_THROW_ON_ERROR), true, 512, JSON_THROW_ON_ERROR)))
                            . ')';
                    }
                    break;

                case 'string':
                    if (str_ends_with($value, '::class')) {
                        $cell .= $value;
                    } elseif (ClassNameAssembler::isFullClassName($value)) {
                        $cell .= '\\' . trim($value, '\\') . '::class';
                    } else {
                        $cell .= '"' . addcslashes($value, '"') . '"';
                    }
                    break;

                case 'NULL':
                    $cell .= 'null';
                    break;
            }

            $parts[] = $cell;
        }

        return $parts;
    }

    /**
     * @throws JsonException
     */
    protected function buildAnnotationArray(array $arguments): array
    {
        $parts = [];
        $isList = array_is_list($arguments);
        foreach ($arguments as $name => $value) {
            if ($isList && is_array($value) && !empty($value['_'])) {
                $name = $value['_'];
                unset($value['_']);
            }

            $cell = '';

            if (!$isList) {
                $cell = $name . '=';
            }

            switch (gettype($value)) {
                case 'integer':
                case 'double':
                    $cell .= $value;
                    break;

                case 'boolean':
                    $cell .= $value ? 'true' : 'false';
                    break;

                case 'array':
                    $cell .= '[' . implode(', ', $this->buildArray($value)) . ']';
                    break;

                case 'object':
                    if ($value instanceof Attribute) {
                        $cell .= '@' . $value->getClassName() . '('
                            . implode(', ', $this->buildAnnotationObjectArgumentsList($value->getArguments()))
                            . ')';
                    } elseif ($value instanceof Annotation) {
                        $cell .= '@' . $value->name . '('
                            . implode(', ', $this->buildAnnotationObjectArgumentsList($value->arguments))
                            . ')';
                    } else {
                        $cell .= '@' . get_class($value) . '('
                            . implode(', ', $this->buildAnnotationObjectArgumentsList(json_decode(json_encode($value, JSON_THROW_ON_ERROR), true, 512, JSON_THROW_ON_ERROR)))
                            . ')';
                    }
                    break;

                case 'string':
                    if (str_ends_with($value, '::class')) {
                        $cell .= '"' . substr($value, 0, -7) . '"';
                    } else {
                        $cell .= '"' . addcslashes($value, '"') . '"';
                    }
                    break;

                case 'NULL':
                    $cell .= 'null';
                    break;
            }

            $parts[] = $cell;
        }

        return $parts;
    }

    /**
     * @throws JsonException
     */
    protected function buildEntity(AbstractEntity $entity): array
    {
        return [
            'namespace'      => ClassNameAssembler::assembleNamespace($entity->className),
            'classShortName' => ClassNameAssembler::assembleShortName($entity->className),
            'uses'           => $entity->uses,
            'annotations'    => $this->buildAnnotations($entity->annotations),
            'extends'        => $entity->extends,
            'interfaces'     => $entity->interfaces,
            'traits'         => $entity->traits,
            'fields'         => [],
            'getters'        => [],
            'setters'        => [],
            'adders'         => [],
        ];
    }

    /**
     * @throws JsonException
     */
    protected function buildField(Field|EntityField|DoctrineField $field): array
    {
        return [
            'name'        => $field->name,
            'annotations' => $this->buildAnnotations($field->annotations),
            'var'         => $field->type,
            'type_hint'   => $this->detectTypeHint($field->type),
            'nullable'    => $field->nullable,
            'is_array'    => $this->isArray($field->type),
        ];
    }

    protected function buildGetter(Field|EntityField|DoctrineField $field): array
    {
        return [
            'name'                    => MethodNameAssembler::assembleGetterName($field->name),
            'display_return_docblock' => $this->isArrayItemSpecified($field->type),
            'type'                    => $field->type,
            'return'                  => $this->detectTypeHint($field->type),
            'nullable'                => $field->nullable,
            'property'                => $field->name,
        ];
    }

    protected function buildSetter(Field|EntityField|DoctrineField $field, string $classShortName): array
    {
        return [
            'name'                   => MethodNameAssembler::assembleSetterName($field->name),
            'display_param_docblock' => $this->isArrayItemSpecified($field->type),
            'type'                   => $field->type,
            'param'                  => $this->detectTypeHint($field->type),
            'return'                 => $classShortName,
            'nullable'               => $field->nullable,
            'property'               => $field->name,
        ];
    }

    protected function buildAdder(string $name, string $type, string $classShortName): array
    {
        return [
            'name'           => MethodNameAssembler::assembleAdderName($name),
            'item_type_hint' => $this->detectItemTypeHint($type),
            'property'       => $name,
            'return'         => $classShortName,
        ];
    }

    /**
     * @throws JsonException
     */
    protected function buildAnnotations(Annotations $annotations): array
    {
        $result = [];
        /** @var Annotation $annotation */
        foreach ($annotations as $annotation) {
            $result[] = [
                'name'      => ltrim($annotation->name, '@'),
                'arguments' => $this->useAttributes
                    ? $this->buildAttributeObjectArgumentsList($annotation->arguments)
                    : $this->buildAnnotationObjectArgumentsList($annotation->arguments),
            ];
        }

        return $result;
    }

    protected function detectTypeHint(string $type): string
    {
        if ($this->isArrayItemSpecified($type)) {
            $type = 'array';
        }

        return match ($type) {
            'integer' => 'int',
            'boolean' => 'bool',
            default   => $type,
        };
    }

    protected function detectItemTypeHint(string $type): string
    {
        if ($this->isArrayItemSpecified($type)) {
            return $this->detectTypeHint(substr($type, 0, -2));
        }

        return $type;
    }

    protected function isArray(string $type): bool
    {
        return $type === 'array' || $this->isArrayItemSpecified($type);
    }

    protected function isArrayItemSpecified(string $type): bool
    {
        return str_ends_with($type, '[]');
    }
}
