<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Builder;

use XCart\Bundle\DTOGeneratorBundle\DataProvider\DoctrineEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Transformer;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\TransformersDataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicResponseType;
use XCart\Bundle\DTOGeneratorBundle\Enum\TransformerType;
use XCart\Bundle\DTOGeneratorBundle\Service\NameResolverInterface;

class LogicTransformersBuilder implements BuilderInterface
{
    public function __construct(
        private DataProvider $dataProvider,
        private TransformersDataProvider $transformersDataProvider,
        private NameResolverInterface $nameResolver,
        private DoctrineEntitiesDataProvider $doctrineEntitiesDataProvider,
    ) {
    }

    public function build(): void
    {
        /** @var Entity $entity */
        foreach ($this->dataProvider as $entity) {
            if (!$entity->doctrineEntity) {
                continue;
            }

            if ($this->doctrineEntitiesDataProvider->hasClassName($entity->doctrineEntity)) {
                $this->addTransformer($entity->doctrineEntity, $entity->className);
                $this->addTransformer($entity->className, $entity->doctrineEntity);
            }

            if ($entity->new) {
                continue;
            }

            if ($entity->generateGetOne) {
                $this->addTransformer(
                    $entity->doctrineEntity,
                    $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::GET_ONE()),
                    TransformerType::LOGIC_RESPONSE_ITEM_PAYLOAD(),
                );
            }

            if ($entity->generateGetCollection) {
                $this->addTransformer(
                    $entity->doctrineEntity,
                    $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::GET_COLLECTION()),
                    TransformerType::LOGIC_RESPONSE_COLLECTION_PAYLOAD(),
                );
            }

            if ($entity->generateCreateOne) {
                $this->addTransformer(
                    $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::CREATE_ONE()),
                    $entity->doctrineEntity,
                    TransformerType::LOGIC_REQUEST_ITEM_PAYLOAD(),
                );

                $this->addTransformer(
                    $entity->doctrineEntity,
                    $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::CREATE_ONE()),
                    TransformerType::LOGIC_RESPONSE_ITEM_PAYLOAD(),
                );
            }

            if ($entity->generateUpdateOne) {
                $this->addTransformer(
                    $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::UPDATE_ONE()),
                    $entity->doctrineEntity,
                    TransformerType::LOGIC_REQUEST_ITEM_PAYLOAD(),
                );

                $this->addTransformer(
                    $entity->doctrineEntity,
                    $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::UPDATE_ONE()),
                    TransformerType::LOGIC_RESPONSE_ITEM_PAYLOAD(),
                );
            }
        }
    }

    private function addTransformer(string $from, string $to, ?TransformerType $type = null): void
    {
        $transformer = new Transformer();
        $transformer->from = $from;
        $transformer->to = $to;

        if ($type) {
            $transformer->type = $type;
        }

        $this->transformersDataProvider->add($transformer);
    }
}
