<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service\Builder;

use XCart\Bundle\DTOGeneratorBundle\Enum\DTOType;
use XCart\Bundle\LogicBundle\DTO\Request\FilterOwnerRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Request\OrderRuleOwnerRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Request\PaginatorRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Request\PaginatorRequestTrait;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DTO;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\EntityField;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicRequestEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\LogicResponseEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTODataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicRequestEntitiesDataProvider as RequestDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicResponseEntitiesDataProvider as ResponseDataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicResponseType;
use XCart\Bundle\DTOGeneratorBundle\Service\NameResolverInterface;
use XCart\Bundle\LogicBundle\DTO\Response\PageOwnerInterface;
use XCart\Bundle\LogicBundle\DTO\Response\PageOwnerTrait;

class LogicRequestResponseBuilder implements BuilderInterface
{
    public function __construct(
        private DataProvider $dataProvider,
        private RequestDataProvider $requestDataProvider,
        private ResponseDataProvider $responseDataProvider,
        private DTODataProvider $DTODataProvider,
        private NameResolverInterface $nameResolver,
    ) {
    }

    public function build(): void
    {
        /** @var Entity $entity */
        foreach ($this->dataProvider as $entity) {
            if ($entity->new) {
                continue;
            }

            if ($entity->generateGetOne) {
                $this->addGetOneEntities($entity, $this->requestDataProvider, $this->responseDataProvider);
            }

            if ($entity->generateGetCollection) {
                $this->addGetCollectionEntities($entity, $this->requestDataProvider, $this->responseDataProvider, $this->DTODataProvider);
            }

            if ($entity->generateCreateOne) {
                $this->addCreateOneEntities($entity, $this->requestDataProvider, $this->responseDataProvider);
            }

            if ($entity->generateUpdateOne) {
                $this->addUpdateOneEntities($entity, $this->requestDataProvider, $this->responseDataProvider);
            }

            if ($entity->generateDeleteOne) {
                $this->addDeleteOneEntities($entity, $this->requestDataProvider, $this->responseDataProvider);
            }
        }
    }

    private function addGetOneEntities(Entity $entity, RequestDataProvider $requestDataProvider, ResponseDataProvider $responseDataProvider): void
    {
        $className = $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::GET_ONE());
        if (!$requestDataProvider->hasClassName($className)) {
            $request = new LogicRequestEntity();
            $request->className = $className;
            $request->type = LogicRequestType::GET_ONE();
            $request->entity = $entity->className;
            $requestDataProvider->add($request);
        }

        $className = $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::GET_ONE());
        if (!$responseDataProvider->hasClassName($className)) {
            $response = new LogicResponseEntity();
            $response->className = $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::GET_ONE());
            $response->type = LogicResponseType::GET_ONE();
            $response->entity = $entity->className;
            $responseDataProvider->add($response);
        }
    }

    private function addGetCollectionEntities(Entity $entity, RequestDataProvider $requestDataProvider, ResponseDataProvider $responseDataProvider, DTODataProvider $DTODataProvider): void
    {
        $className = $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::GET_COLLECTION());
        if (!$requestDataProvider->hasClassName($className)) {
            $request = new LogicRequestEntity();
            $request->className = $className;
            $requestDataProvider->add($request);

        } else {
            /** @var LogicRequestEntity $request */
            $request = $requestDataProvider->getByClassName($className);
        }

        $request->type = LogicRequestType::GET_COLLECTION();
        $request->entity = $request->entity ?? $entity->className;
        $request->traits[] = PaginatorRequestTrait::class;
        $request->interfaces[] = PaginatorRequestInterface::class;
        $request->interfaces[] = FilterOwnerRequestInterface::class;
        $request->interfaces[] = OrderRuleOwnerRequestInterface::class;

        // Add / detect filter sub DTO (logic layer)
        $filterClassName = $this->nameResolver->buildLogicEntityFilterClassName($entity->className);
        $filter = $DTODataProvider->getByClassName($filterClassName);

        if (!$filter) {
            $filter = new DTO();
            $filter->className = $filterClassName;
            $filter->type = DTOType::LOGIC_REQUEST_GET_LIST_FILTER();
            $DTODataProvider->add($filter);
        }

        $request->fields['filter'] = new EntityField();
        $request->fields['filter']->name = 'filter';
        $request->fields['filter']->type = $filterClassName;

        // Add / detect order rule sub DTO (logic layer)
        $orderRuleClassName = $this->nameResolver->buildLogicEntityOrderRuleClassName($entity->className);
        $orderRule = $DTODataProvider->getByClassName($orderRuleClassName);

        if (!$orderRule) {
            $orderRule = new DTO();
            $orderRule->className = $orderRuleClassName;
            $orderRule->type = DTOType::LOGIC_REQUEST_GET_LIST_ORDER_RULE();
            $DTODataProvider->add($orderRule);
        }

        $request->fields['orderRule'] = new EntityField();
        $request->fields['orderRule']->name = 'orderRule';
        $request->fields['orderRule']->type = $orderRuleClassName;

        $className = $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::GET_COLLECTION());
        if (!$responseDataProvider->hasClassName($className)) {
            $response = new LogicResponseEntity();
            $response->className = $className;
            $response->type = LogicResponseType::GET_COLLECTION();
            $response->entity = $entity->className;
            $response->interfaces[] = PageOwnerInterface::class;
            $response->traits[] = PageOwnerTrait::class;

            $responseDataProvider->add($response);
        }
    }

    private function addCreateOneEntities(Entity $entity, RequestDataProvider $requestDataProvider, ResponseDataProvider $responseDataProvider): void
    {
        $className = $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::CREATE_ONE());
        if (!$requestDataProvider->hasClassName($className)) {
            $request = new LogicRequestEntity();
            $request->className = $className;
            $request->type = LogicRequestType::CREATE_ONE();
            $request->entity = $this->nameResolver->buildNewEntityClassName($entity->className);
            $requestDataProvider->add($request);
        }

        $className = $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::CREATE_ONE());
        if (!$responseDataProvider->hasClassName($className)) {
            $response = new LogicResponseEntity();
            $response->className = $className;
            $response->type = LogicResponseType::CREATE_ONE();
            $response->entity = $this->nameResolver->buildNewEntityClassName($entity->className);
            $responseDataProvider->add($response);
        }
    }

    private function addUpdateOneEntities(Entity $entity, RequestDataProvider $requestDataProvider, ResponseDataProvider $responseDataProvider): void
    {
        $className = $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::UPDATE_ONE());
        if (!$requestDataProvider->hasClassName($className)) {
            $request = new LogicRequestEntity();
            $request->className = $className;
            $request->type = LogicRequestType::UPDATE_ONE();
            $request->entity = $entity->className;
            $requestDataProvider->add($request);
        }

        $className = $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::UPDATE_ONE());
        if (!$responseDataProvider->hasClassName($className)) {
            $response = new LogicResponseEntity();
            $response->className = $className;
            $response->type = LogicResponseType::UPDATE_ONE();
            $response->entity = $entity->className;
            $responseDataProvider->add($response);
        }
    }

    private function addDeleteOneEntities(Entity $entity, RequestDataProvider $requestDataProvider, ResponseDataProvider $responseDataProvider): void
    {
        $className = $this->nameResolver->buildLogicRequestClassName($entity->className, LogicRequestType::DELETE_ONE());
        if (!$requestDataProvider->hasClassName($className)) {
            $request = new LogicRequestEntity();
            $request->className = $className;
            $request->type = LogicRequestType::DELETE_ONE();
            $request->entity = $entity->className;
            $requestDataProvider->add($request);
        }

        $className = $this->nameResolver->buildLogicResponseClassName($entity->className, LogicResponseType::DELETE_ONE());
        if (!$responseDataProvider->hasClassName($className)) {
            $response = new LogicResponseEntity();
            $response->className = $className;
            $response->type = LogicResponseType::DELETE_ONE();
            $response->entity = $entity->className;
            $responseDataProvider->add($response);
        }
    }
}
