<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\Service;

use ApiPlatform\Core\Metadata\Resource\Factory\ResourceMetadataFactoryInterface;
use InvalidArgumentException;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\DataPersister;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Persister\Assembler\PayloadAssembler as APICreateOneRequestAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Persister\Assembler\PayloadResponseAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Persister\Persister;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Remover\Assembler\IdBasedAssembler as APIRequestIdBasedAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Remover\Remover;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\Filter\FilterAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\OrderRule\OrderRuleAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\PayloadResponseAssembler as APIGetCollectionResponseAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\RequestAssembler as APIGetCollectionRequestAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\DataProvider as APICollectionDataProvider;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Enricher\Pagination\PaginationEnricher;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Item\Assembler\PayloadResponseAssembler as APIGetOnePayloadResponseAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Item\Assembler\RequestAssembler as APIGetOneRequestAssembler;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Item\DataProvider as APIItemDataProvider;
use XCart\Bundle\APIPlatformBridgeBundle\Extension\FilterExtensionInterface;
use XCart\Bundle\APIPlatformBridgeBundle\Extension\OrderRuleExtensionInterface;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\APIEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\APIEntity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTODataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\LogicEntitiesDataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\TransformersDataProvider;
use XCart\Bundle\DTOGeneratorBundle\Enum\LogicRequestType;

class APICRUDRegister
{
    public function __construct(
        private NameResolverInterface     $nameResolver,
        private APIEntitiesDataProvider   $dataProvider,
        private TransformersDataProvider  $transformersDataProvider,
        private LogicEntitiesDataProvider $logicEntitiesDataProvider,
        private DTODataProvider           $DTODataProvider,
    )
    {
    }

    public function register(ContainerBuilder $container): void
    {
        /** @var APIEntity $entity */
        foreach ($this->dataProvider as $entity) {
            if ($entity->registerGetOne) {
                $this->registerGetOne($container, $entity);
            }

            if ($entity->registerGetCollection) {
                $this->registerGetCollection($container, $entity);
            }

            if ($entity->registerCreateOne || $entity->registerUpdateOne || $entity->registerFullUpdateOne || $entity->registerDeleteOne) {
                $this->registerPersister($container, $entity);
            }

            if ($entity->registerCreateOne) {
                $this->registerCreateOne($container, $entity);
            }

            if ($entity->registerUpdateOne || $entity->registerFullUpdateOne) {
                $this->registerUpdateOne($container, $entity);
            }

            if ($entity->registerDeleteOne) {
                $this->registerDeleteOne($container, $entity);
            }
        }
    }

    private function registerGetOne(ContainerBuilder $container, APIEntity $entity): void
    {
        if (!$entity->logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Logic entity not declared for API entity "%s"', $entity->className)
            );
        }

        $logicEntity = $this->logicEntitiesDataProvider->getByClassName($entity->logicEntity);
        if (!$logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Cannot find logic entity "%s"', $entity->logicEntity)
            );
        }

        // data provider
        $getOneDataProviderServiceId = $this->nameResolver->buildAPIGetOneDataProviderId($entity->className);
        if (!$container->hasDefinition($getOneDataProviderServiceId)) {
            $definition = (new Definition(APIItemDataProvider::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->setArgument('$classes', [$entity->className])
                ->setArgument('$action', new Reference($this->nameResolver->buildLogicGetOneId($logicEntity->className)))
                ->setArgument('$requestAssembler', new Reference($this->nameResolver->buildAPIGetOneRequestAssemblerId($entity->className)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildAPIGetOneResponseAssemblerId($entity->className)));

            $container->setDefinition($getOneDataProviderServiceId, $definition);
        }

        // request assembler
        $getOneRequestAssemblerServiceId = $this->nameResolver->buildAPIGetOneRequestAssemblerId($entity->className);
        if (!$container->hasDefinition($getOneRequestAssemblerServiceId)) {
            $requestClassName = $this->nameResolver->buildLogicRequestClassName($logicEntity->className, LogicRequestType::GET_ONE());
            $definition = (new Definition(APIGetOneRequestAssembler::class))
                ->setArgument('$requestFactory', new Reference($this->nameResolver->buildFactoryInterfaceName($requestClassName)));

            $container->setDefinition($getOneRequestAssemblerServiceId, $definition);
        }

        // response assembler
        $getOneResponseAssemblerServiceId = $this->nameResolver->buildAPIGetOneResponseAssemblerId($entity->className);
        if (!$container->hasDefinition($getOneResponseAssemblerServiceId)) {
            $transformer = $this->transformersDataProvider->getByFromTo($logicEntity->className, $entity->className);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic entity "%s" to API entity "%s"',
                        $logicEntity->className,
                        $entity->className,
                    )
                );
            }

            $definition = (new Definition(APIGetOnePayloadResponseAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($getOneResponseAssemblerServiceId, $definition);
        }
    }

    private function registerGetCollection(ContainerBuilder $container, APIEntity $entity): void
    {
        if (!$entity->logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Logic entity not declared for API entity "%s"', $entity->className)
            );
        }

        $logicEntity = $this->logicEntitiesDataProvider->getByClassName($entity->logicEntity);
        if (!$logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Cannot find logic entity "%s"', $entity->logicEntity)
            );
        }

        $filterClassName = $this->nameResolver->buildLogicEntityFilterClassName($logicEntity->className);
        if (!$this->DTODataProvider->hasClassName($filterClassName)) {
            $filterClassName = null;
        }

        $orderRuleClassName = $this->nameResolver->buildLogicEntityOrderRuleClassName($logicEntity->className);
        if (!$this->DTODataProvider->hasClassName($orderRuleClassName)) {
            $orderRuleClassName = null;
        }

        // data provider
        $getCollectionDataProviderServiceId = $this->nameResolver->buildAPIGetCollectionDataProviderId($entity->className);
        if (!$container->hasDefinition($getCollectionDataProviderServiceId)) {
            $definition = (new Definition(APICollectionDataProvider::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->setArgument('$classes', [$entity->className])
                ->setArgument('$action', new Reference($this->nameResolver->buildLogicGetCollectionId($logicEntity->className)))
                ->setArgument('$requestAssembler', new Reference($this->nameResolver->buildAPIGetCollectionRequestAssemblerId($entity->className)))
                ->setArgument('$responseAssembler', new Reference($this->nameResolver->buildAPIGetCollectionResponseAssemblerId($entity->className)));

            $container->setDefinition($getCollectionDataProviderServiceId, $definition);
        }

        // request assembler
        $getCollectionRequestAssemblerServiceId = $this->nameResolver->buildAPIGetCollectionRequestAssemblerId($entity->className);
        if (!$container->hasDefinition($getCollectionRequestAssemblerServiceId)) {
            $requestClassName = $this->nameResolver->buildLogicRequestClassName($logicEntity->className, LogicRequestType::GET_COLLECTION());
            $definition = (new Definition(APIGetCollectionRequestAssembler::class))
                ->setArgument('$requestFactory', new Reference($this->nameResolver->buildFactoryInterfaceName($requestClassName)))
                ->setArgument('$paginationEnricher', new Reference($this->nameResolver->buildAPIGetCollectionRequestPaginationEnricherId($entity->className)));

            if ($filterClassName) {
                $definition
                    ->setArgument('$filterAssembler', new Reference($this->nameResolver->buildAPIGetCollectionRequestFilterAssemblerId($entity->className)));
            }

            if ($orderRuleClassName) {
                $definition
                    ->setArgument('$orderRuleAssembler', new Reference($this->nameResolver->buildAPIGetCollectionRequestOrderRuleAssemblerId($entity->className)));
            }

            $container->setDefinition($getCollectionRequestAssemblerServiceId, $definition);
        }

        // pagination enricher
        $getCollectionRequestPaginationEnricherServiceId = $this->nameResolver->buildAPIGetCollectionRequestPaginationEnricherId($entity->className);
        if (!$container->hasDefinition($getCollectionRequestPaginationEnricherServiceId)) {
            $definition = (new Definition(PaginationEnricher::class))
                ->setArgument('$resourceMetadataFactory', new Reference(ResourceMetadataFactoryInterface::class));

            $container->setDefinition($getCollectionRequestPaginationEnricherServiceId, $definition);
        }

        // request filter assembler
        if ($filterClassName && !$container->hasDefinition($this->nameResolver->buildAPIGetCollectionRequestFilterAssemblerId($entity->className))) {
            $definition = (new Definition(FilterAssembler::class))
                ->setArgument('$filterFactory', new Reference($this->nameResolver->buildFactoryInterfaceName($filterClassName)))
                ->setArgument('$filterExtension', new Reference(FilterExtensionInterface::class));

            $container->setDefinition($this->nameResolver->buildAPIGetCollectionRequestFilterAssemblerId($entity->className), $definition);
        }

        // request order rule assembler
        if ($orderRuleClassName && !$container->hasDefinition($this->nameResolver->buildAPIGetCollectionRequestOrderRuleAssemblerId($entity->className))) {
            $orderRuleClassName = $this->nameResolver->buildLogicEntityOrderRuleClassName($logicEntity->className);
            $definition = (new Definition(OrderRuleAssembler::class))
                ->setArgument('$factory', new Reference($this->nameResolver->buildFactoryInterfaceName($orderRuleClassName)))
                ->setArgument('$extension', new Reference(OrderRuleExtensionInterface::class));

            $container->setDefinition($this->nameResolver->buildAPIGetCollectionRequestOrderRuleAssemblerId($entity->className), $definition);
        }

        // response assembler
        $getCollectionResponseAssemblerServiceId = $this->nameResolver->buildAPIGetCollectionResponseAssemblerId($entity->className);
        if (!$container->hasDefinition($getCollectionResponseAssemblerServiceId)) {
            $transformer = $this->transformersDataProvider->getByFromTo($logicEntity->className, $entity->className);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic entity "%s" to API entity "%s"',
                        $logicEntity->className,
                        $entity->className,
                    )
                );
            }

            $definition = (new Definition(APIGetCollectionResponseAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($getCollectionResponseAssemblerServiceId, $definition);
        }
    }

    private function registerPersister(ContainerBuilder $container, APIEntity $entity): void
    {
        if (!$entity->logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Logic entity not declared for API entity "%s"', $entity->className)
            );
        }

        $logicEntity = $this->logicEntitiesDataProvider->getByClassName($entity->logicEntity);
        if (!$logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Cannot find logic entity "%s"', $entity->logicEntity)
            );
        }

        // data persister
        $dataPersisterServiceId = $this->nameResolver->buildAPIDataPersisterId($entity->className);
        if (!$container->hasDefinition($dataPersisterServiceId)) {
            $definition = (new Definition(DataPersister::class))
                ->setAutowired(true)
                ->setAutoconfigured(true)
                ->setArgument('$classes', [$entity->className])
                ->setArgument('$persister', new Reference($this->nameResolver->buildAPIPersisterId($entity->className), $container::NULL_ON_INVALID_REFERENCE))
                ->setArgument('$remover', new Reference($this->nameResolver->buildAPIRemoverId($entity->className), $container::NULL_ON_INVALID_REFERENCE));

            $container->setDefinition($dataPersisterServiceId, $definition);
        }

        // persister
        if ($entity->registerCreateOne && !$container->hasDefinition($this->nameResolver->buildAPIPersisterId($entity->className))) {
            $definition = (new Definition(Persister::class))
                ->setArgument('$createAction', new Reference($this->nameResolver->buildLogicCreateOneId($logicEntity->className), $container::NULL_ON_INVALID_REFERENCE))
                ->setArgument('$updateAction', new Reference($this->nameResolver->buildLogicUpdateOneId($logicEntity->className), $container::NULL_ON_INVALID_REFERENCE))
                ->setArgument('$createRequestAssembler', new Reference($this->nameResolver->buildAPIPersisterCreateRequestAssemblerId($entity->className), $container::NULL_ON_INVALID_REFERENCE))
                ->setArgument('$updateRequestAssembler', new Reference($this->nameResolver->buildAPIPersisterUpdateRequestAssemblerId($entity->className), $container::NULL_ON_INVALID_REFERENCE))
                ->setArgument('$createResponseAssembler', new Reference($this->nameResolver->buildAPIPersisterCreateResponseAssemblerId($entity->className)))
                ->setArgument('$updateResponseAssembler', new Reference($this->nameResolver->buildAPIPersisterUpdateResponseAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildAPIPersisterId($entity->className), $definition);
        }

        // remover
        if ($entity->registerDeleteOne && !$container->hasDefinition($this->nameResolver->buildAPIRemoverId($entity->className))) {
            $definition = (new Definition(Remover::class))
                ->setArgument('$action', new Reference($this->nameResolver->buildLogicDeleteOneId($logicEntity->className)))
                ->setArgument('$requestAssembler', new Reference($this->nameResolver->buildAPIRemoverRequestAssemblerId($entity->className)));

            $container->setDefinition($this->nameResolver->buildAPIRemoverId($entity->className), $definition);
        }

        // create response assembler
        $persisterCreateResponseAssemblerServiceId = $this->nameResolver->buildAPIPersisterCreateResponseAssemblerId($entity->className);
        if (!$container->hasDefinition($persisterCreateResponseAssemblerServiceId)) {
            $transformer = $this->transformersDataProvider->getByFromTo($this->nameResolver->buildNewEntityClassName($logicEntity->className), $entity->className);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic entity "%s" to API entity "%s"',
                        $logicEntity->className,
                        $entity->className,
                    )
                );
            }

            $definition = (new Definition(PayloadResponseAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($persisterCreateResponseAssemblerServiceId, $definition);
        }

        // create response assembler
        $persisterUpdateResponseAssemblerServiceId = $this->nameResolver->buildAPIPersisterUpdateResponseAssemblerId($entity->className);
        if (!$container->hasDefinition($persisterUpdateResponseAssemblerServiceId)) {
            $transformer = $this->transformersDataProvider->getByFromTo($logicEntity->className, $entity->className);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic entity "%s" to API entity "%s"',
                        $logicEntity->className,
                        $entity->className,
                    )
                );
            }

            $definition = (new Definition(PayloadResponseAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)));

            $container->setDefinition($persisterUpdateResponseAssemblerServiceId, $definition);
        }
    }

    private function registerCreateOne(ContainerBuilder $container, APIEntity $entity): void
    {
        if (!$entity->logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Logic entity not declared for API entity "%s"', $entity->className)
            );
        }

        $logicEntity = $this->logicEntitiesDataProvider->getByClassName($entity->logicEntity);
        if (!$logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Cannot find logic entity "%s"', $entity->logicEntity)
            );
        }

        // request assembler
        $persisterCreateRequestAssemblerServiceId = $this->nameResolver->buildAPIPersisterCreateRequestAssemblerId($entity->className);
        if (!$container->hasDefinition($persisterCreateRequestAssemblerServiceId)) {
            $transformer = $this->transformersDataProvider->getByFromTo($entity->className, $this->nameResolver->buildNewEntityClassName($logicEntity->className));
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic entity "%s" to API entity "%s"',
                        $entity->className,
                        $this->nameResolver->buildNewEntityClassName($logicEntity->className),
                    )
                );
            }
            $requestClassName = $this->nameResolver->buildLogicRequestClassName($logicEntity->className, LogicRequestType::CREATE_ONE());
            $definition = (new Definition(APICreateOneRequestAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)))
                ->setArgument('$factory', new Reference($this->nameResolver->buildFactoryInterfaceName($requestClassName)));

            $container->setDefinition($persisterCreateRequestAssemblerServiceId, $definition);
        }
    }

    private function registerUpdateOne(ContainerBuilder $container, APIEntity $entity): void
    {
        if (!$entity->logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Logic entity not declared for API entity "%s"', $entity->className)
            );
        }

        $logicEntity = $this->logicEntitiesDataProvider->getByClassName($entity->logicEntity);
        if (!$logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Cannot find logic entity "%s"', $entity->logicEntity)
            );
        }

        // request assembler
        $persisterUpdateRequestAssemblerServiceId = $this->nameResolver->buildAPIPersisterUpdateRequestAssemblerId($entity->className);
        if (!$container->hasDefinition($persisterUpdateRequestAssemblerServiceId)) {
            $transformer = $this->transformersDataProvider->getByFromTo($entity->className, $logicEntity->className);
            if (!$transformer) {
                throw new InvalidArgumentException(
                    sprintf(
                        'Cannot find transformer from logic entity "%s" to API entity "%s"',
                        $entity->className,
                        $this->nameResolver->buildNewEntityClassName($logicEntity->className),
                    )
                );
            }
            $requestClassName = $this->nameResolver->buildLogicRequestClassName($logicEntity->className, LogicRequestType::UPDATE_ONE());
            $definition = (new Definition(APICreateOneRequestAssembler::class))
                ->setArgument('$transformer', new Reference($this->nameResolver->buildTransformerInterfaceName($transformer)))
                ->setArgument('$factory', new Reference($this->nameResolver->buildFactoryInterfaceName($requestClassName)));

            $container->setDefinition($persisterUpdateRequestAssemblerServiceId, $definition);
        }
    }

    private function registerDeleteOne(ContainerBuilder $container, APIEntity $entity): void
    {
        if (!$entity->logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Logic entity not declared for API entity "%s"', $entity->className)
            );
        }

        $logicEntity = $this->logicEntitiesDataProvider->getByClassName($entity->logicEntity);
        if (!$logicEntity) {
            throw new InvalidArgumentException(
                sprintf('Cannot find logic entity "%s"', $entity->logicEntity)
            );
        }

        // request assembler
        $removerRequestAssemblerServiceId = $this->nameResolver->buildAPIRemoverRequestAssemblerId($entity->className);
        if (!$container->hasDefinition($removerRequestAssemblerServiceId)) {
            $requestClassName = $this->nameResolver->buildLogicRequestClassName($logicEntity->className, LogicRequestType::DELETE_ONE());
            $definition = (new Definition(APIRequestIdBasedAssembler::class))
                ->setArgument('$factory', new Reference($this->nameResolver->buildFactoryInterfaceName($requestClassName)));

            $container->setDefinition($removerRequestAssemblerServiceId, $definition);
        }
    }
}
