<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\DependencyInjection\Compiler;

use InvalidArgumentException;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\DoctrineBridgeBundle\Entity\EntityInterface;
use XCart\Bundle\DoctrineBridgeBundle\QueryBuilder\CommonQueryBuilder;
use XCart\Bundle\DoctrineBridgeBundle\Repository\GeneratedRepositoryInterface;
use XCart\Bundle\DoctrineBridgeBundle\Repository\WriteRepository;
use XCart\Bundle\DTOGeneratorBundle\Service\NameResolverInterface;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DoctrineEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineEntity as Entity;

class UpdateGeneratedEntitiesPass implements CompilerPassInterface
{
    public function process(ContainerBuilder $container): void
    {
        if (!$container->hasDefinition('xcart.dto_generator.data_provider.doctrine_entities')) {
            return;
        }

        /** @var NameResolverInterface $nameResolver */
        $nameResolver = $container->get(NameResolverInterface::class);

        /** @var DataProvider $dataProvider */
        $dataProvider = $container->get('xcart.dto_generator.data_provider.doctrine_entities');
        /** @var Entity $entity */
        foreach ($dataProvider as $entity) {
            if (!$entity->table) {
                throw new InvalidArgumentException(
                    sprintf('Entity "%s" is Doctrine entity but table property not specified', $entity->className)
                );
            }

            if (!$entity->annotations->hasAnnotation('ORM\Entity')) {
                $entity->annotations->addAnnotationFromArray(
                    [
                        'repositoryClass' => $nameResolver->buildRepositoryClassName($entity->className)
                    ],
                    'ORM\Entity'
                );
            }

            $entity->uses[] = 'Doctrine\ORM\Mapping as ORM';
            $entity->interfaces[] = EntityInterface::class;

            $entity->repository->interfaces[] = GeneratedRepositoryInterface::class;

            if (!$entity->annotations->hasAnnotation('ORM\Table')) {
                $entity->annotations->addAnnotationFromArray(['name' => $entity->table], 'ORM\Table');
            }

            if (!$entity->repository->extends) {
                $entity->repository->extends = WriteRepository::class;
            }

            $entity->repository->uses[] = $entity->repository->extends;
            $entity->repository->extends = ClassNameAssembler::assembleShortName($entity->repository->extends);

            foreach ($entity->repository->traits as $i => $trait) {
                $entity->repository->uses[] = $trait;
                $entity->repository->traits[$i] = ClassNameAssembler::assembleShortName($trait);
            }

            foreach ($entity->repository->interfaces as $i => $interface) {
                $entity->repository->uses[] = $interface;
                $entity->repository->interfaces[$i] = ClassNameAssembler::assembleShortName($interface);
            }

            $entity->repository->interfaces[] = ClassNameAssembler::assembleShortName(
                $nameResolver->buildRepositoryReadInterfaceName($entity->className)
            );
            $entity->repository->interfaces[] = ClassNameAssembler::assembleShortName(
                $nameResolver->buildRepositoryWriteInterfaceName($entity->className)
            );

            foreach ($entity->queryBuilder->interfaces as $i => $interface) {
                $entity->queryBuilder->uses[] = $interface;
                $entity->queryBuilder->interfaces[$i] = ClassNameAssembler::assembleShortName($interface);
            }

            $entity->queryBuilder->interfaces[] = ClassNameAssembler::assembleShortName(
                $nameResolver->buildQueryBuilderInterfaceName($entity->className)
            );

            sort($entity->repository->uses);
            sort($entity->repository->interfaces);
            sort($entity->queryBuilder->interfaces);

            if (!$entity->queryBuilder->extends) {
                $entity->queryBuilder->extends = CommonQueryBuilder::class;
            }

            $entity->queryBuilder->uses[] = $entity->queryBuilder->extends;
            $entity->queryBuilder->extends = ClassNameAssembler::assembleShortName($entity->queryBuilder->extends);
        }
    }
}
