<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\DependencyInjection\Compiler;

use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;
use XCart\Bundle\DoctrineBridgeBundle\DataSource\Decorator\ReadOnly\ReadOnlyDecorator;
use XCart\Bundle\DTOGeneratorBundle\Service\NameResolverInterface;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DoctrineEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\DoctrineEntity as Entity;

class RegisterReadDataSourcesPass implements CompilerPassInterface
{
    private const ENTITY_MANAGER_ID = 'doctrine.orm.entity_manager';
    private const WRITE_ENTITY_MANAGER_ID = 'doctrine.orm.write_entity_manager';
    private const READ_ENTITY_MANAGER_ID = 'doctrine.orm.read_entity_manager';

    public function process(ContainerBuilder $container): void
    {
        if (!$container->hasDefinition('xcart.dto_generator.data_provider.doctrine_entities')) {
            return;
        }

        if ($container->hasDefinition(self::READ_ENTITY_MANAGER_ID)) {
            $entityManagerId = self::READ_ENTITY_MANAGER_ID;
        } elseif ($container->hasDefinition(self::WRITE_ENTITY_MANAGER_ID)) {
            $entityManagerId = self::WRITE_ENTITY_MANAGER_ID;
        } else {
            $entityManagerId = self::ENTITY_MANAGER_ID;
        }

        /** @var NameResolverInterface $nameResolver */
        $nameResolver = $container->get(NameResolverInterface::class);

        /** @var DataProvider $dataProvider */
        $dataProvider = $container->get('xcart.dto_generator.data_provider.doctrine_entities');
        /** @var Entity $entity */
        foreach ($dataProvider as $entity) {
            // Data source
            $className = $nameResolver->buildDataSourceReadClassName($entity->className);
            $definition = (new Definition($className))
                ->setAutowired(true)
                ->setArgument('$repository', new Reference($nameResolver->buildRepositoryReadId($entity->className)))
                ->setArgument('$collectionClassName', $nameResolver->buildCollectionClassName($entity->className));

            $container->setDefinition($className, $definition);

            $container->setAlias($nameResolver->buildDataSourceReadInterfaceName($entity->className), $className);
            $container->setAlias($nameResolver->buildDataSourceReadId($entity->className), $className);

            // Read-only decorator
            $id = $nameResolver->buildDataSourceReadClassName($entity->className) . '\_\Decorator\ReadOnly';
            $definition = (new Definition(ReadOnlyDecorator::class))
                ->setAutowired(true)
                ->setDecoratedService($nameResolver->buildDataSourceReadClassName($entity->className))
                ->setArgument('$inner', new Reference($id . '.inner'))
                ->setArgument('$entityManager', new Reference($entityManagerId));

            $container->setDefinition($id, $definition);
        }
    }
}
