<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DTOGeneratorBundle\DataProvider\Factory;

use XCart\Bundle\CommonBundle\Assembler\ClassNameAssembler;
use XCart\Bundle\DTOGeneratorBundle\Collection\Annotations;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\APIEntitiesDataProvider as DataProvider;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\APIEntity as Entity;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\APIFilter;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\APIOperation;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\Factory;
use XCart\Bundle\DTOGeneratorBundle\DataProvider\DTO\EntityField as Field;

class APIEntitiesDataProviderFactory
{
    public function __invoke(array $config): DataProvider
    {
        return $this->buildDataProvider($config);
    }

    private function buildDataProvider(array $config): DataProvider
    {
        $dataProvider = new DataProvider();

        if (empty($config['entities']) || empty($config['entities']['api'])) {
            return $dataProvider;
        }

        foreach ($config['entities']['api'] as $className => $entity) {
            $object = new Entity();
            $object->annotations = Annotations::buildFromArray($entity['annotations'] ?? []);
            $object->className = $className;
            $object->idType = $entity['idType'];
            $object->traits = $entity['traits'] ?? [];
            $object->interfaces = $entity['interfaces'] ?? [];
            $object->logicEntity = $entity['logicEntity'] ?? null;
            $object->path = $entity['path'];
            $object->name = $entity['name'] ?? ClassNameAssembler::assembleShortName($object->className);

            $object->registerGetOne = $entity['register_get_one'] ?? false;
            $object->registerGetCollection = $entity['register_get_collection'] ?? false;
            $object->registerCreateOne = $entity['register_create_one'] ?? false;
            $object->registerUpdateOne = $entity['register_update_one'] ?? false;
            $object->registerFullUpdateOne = $entity['register_full_update_one'] ?? false;
            $object->registerDeleteOne = $entity['register_delete_one'] ?? false;

            $object->normalizationContext->groups = $entity['normalizationContext']['groups'] ?? ['read'];
            $object->denormalizationContext->groups = $entity['denormalizationContext']['groups'] ?? ['insert', 'update'];

            $object->itemOperations = [];
            foreach ($entity['itemOperations'] as $name => $row) {
                $operation = new APIOperation();
                $operation->method = !empty($row['method']) ? strtoupper($row['method']) : 'GET';
                $operation->path = $row['path'];
                $operation->identifiers = $row['identifiers'];
                $operation->requirements = $row['requirements'];
                $operation->openapiContext = $row['openapi_context'];
                $operation->normalizationContext->groups = $row['normalization_context']['groups'] ?? [];
                $operation->denormalizationContext->groups = $row['denormalization_context']['groups'] ?? [];

                if (empty($row['normalization_context']['groups'])) {
                    switch ($operation->method) {
                        case 'GET':
                        case 'PUT':
                        case 'PATCH':
                            $operation->normalizationContext->groups = ['read'];
                            break;
                    }
                }

                if (empty($row['denormalization_context']['groups'])) {
                    switch ($operation->method) {
                        case 'PUT':
                        case 'PATCH':
                            $operation->denormalizationContext->groups = ['update'];
                            break;
                    }
                }

                $object->itemOperations[$name] = $operation;
            }

            $object->collectionOperations = [];
            foreach ($entity['collectionOperations'] as $name => $row) {
                $operation = new APIOperation();
                $operation->method = !empty($row['method']) ? strtoupper($row['method']) : 'GET';
                $operation->path = $row['path'];
                $operation->identifiers = $row['identifiers'];
                $operation->requirements = $row['requirements'];
                $operation->openapiContext = $row['openapi_context'];
                $operation->normalizationContext->groups = $row['normalization_context']['groups'] ?? [];
                $operation->denormalizationContext->groups = $row['denormalization_context']['groups'] ?? [];

                if (empty($row['normalization_context']['groups'])) {
                    switch ($operation->method) {
                        case 'GET':
                            $operation->normalizationContext->groups = ['read'];
                            break;
                    }
                }

                if (empty($row['denormalization_context']['groups'])) {
                    switch ($operation->method) {
                        case 'POST':
                            $operation->denormalizationContext->groups = ['insert'];
                            break;
                    }
                }

                $object->collectionOperations[$name] = $operation;
            }

            $object->factory = new Factory();
            $object->factory->interfaces = $entity['factory']['interfaces'] ?? [];

            foreach ($entity['filters'] as $filter) {
                $filterEntity = new APIFilter();
                $filterEntity->className = $filter['className'];
                $filterEntity->properties = $filter['properties'];
                $filterEntity->arguments = $filter['arguments'];
                $filterEntity->operations = $filter['operations'] ?: ['get'];

                $object->filters[] = $filterEntity;
            }

            foreach ($entity['fields'] as $name => $field) {
                $fieldEntity = new Field();
                $fieldEntity->annotations = Annotations::buildFromArray($field['annotations'] ?? []);
                $fieldEntity->type = $field['type'];
                $fieldEntity->name = $name;
                $fieldEntity->nullable = $field['nullable'] ?? false;
                $fieldEntity->id = $field['id'] ?? false;
                $fieldEntity->groups = $field['groups'] ?: ['read', 'insert', 'update'];

                $object->fields[$name] = $fieldEntity;
            }

            $dataProvider->set($object->className, $object);
        }

        return $dataProvider;
    }
}
