<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DoctrineBridgeBundle\QueryBuilder;

use Countable;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Doctrine\ORM\Query\Expr\Func;
use Doctrine\ORM\QueryBuilder;
use Iterator;
use XCart\Bundle\DoctrineBridgeBundle\Entity\EntityInterface;

class CommonQueryBuilder extends QueryBuilder implements Countable, QueryBuilderInterface
{
    /**
     * @var string[]
     */
    protected array $joins = [];

    /**
     * @return EntityInterface[]
     */
    public function getResult(): array
    {
        return $this->getQuery()->getResult();
    }

    public function getArrayResult(): array
    {
        return $this->getQuery()->getArrayResult();
    }

    public function getSingleResult(): ?EntityInterface
    {
        try {
            $entity = $this->setMaxResults(1)->getQuery()->getSingleResult();
        } catch (NonUniqueResultException | NoResultException) {
            return null;
        }

        return $entity;
    }

    public function getSingleScalarResult(): float|int|string|null
    {
        try {
            $scalar = $this->setMaxResults(1)->getQuery()->getSingleScalarResult();
        } catch (NonUniqueResultException | NoResultException) {
            return null;
        }

        return $scalar;
    }

    public function iterate(): iterable
    {
        return $this->getQuery()->toIterable();
    }

    public function execute()
    {
        return $this->getQuery()->execute();
    }

    public function has(): bool
    {
        return $this->count() > 0;
    }

    public function count(): int
    {
        return (int) $this->selectCount()->getSingleScalarResult();
    }

    public function countGroupBySafely(): int
    {
        if (!empty($this->getDQLPart('groupBy'))) {
            $metadata = $this->getEntityManager()
                ->getMetadataFactory()
                ->getMetadataFor($this->getDQLPart('from')[0]->getFrom());
            $idFieldName = $metadata->getIdentifier()[0];

            return (new static($this->getEntityManager()))
                ->from($this->getDQLPart('from')[0]->getFrom(), 'countEntity')
                ->where('countEntity.' . $idFieldName . ' IN (' . $this->getDQL() . ')')
                ->setParameters($this->getParameters())
                ->count();
        }

        return (int) $this->selectCount()->getSingleScalarResult();
    }

    public function getMainAlias(): string
    {
        return $this->getDQLPart('from')[0]->getAlias();
    }


    public function linkInner(string $join, ?string $alias = null, ?string $conditionType = null, ?string $condition = null, ?string $indexBy = null): static
    {
        if (!$alias) {
            [, $alias] = explode('.', $join, 2);
        }

        if (!in_array($alias, $this->joins, true)) {
            $this->innerJoin($join, $alias, $conditionType, $condition, $indexBy);
            $this->joins[] = $alias;
        }

        return $this;
    }

    public function linkLeft(string $join, ?string $alias = null, ?string $conditionType = null, ?string $condition = null, ?string $indexBy = null): static
    {
        if (!$alias) {
            [, $alias] = explode('.', $join, 2);
        }

        if (!in_array($alias, $this->joins, true)) {
            $this->leftJoin($join, $alias, $conditionType, $condition, $indexBy);
            $this->joins[] = $alias;
        }

        return $this;
    }

    public function addInCondition(string $field, $data): static
    {
        $this->andWhere($this->getInExpression($field, $data));

        return $this;
    }

    public function getInExpression(string $field, $data): Func
    {
        if (is_scalar($data)) {
            return $this->expr()->in($field, [$data]);
        }

        if ($data instanceof Iterator) {
            return $this->expr()->in($field, iterator_to_array($data));
        }

        return $this->expr()->in($field, $data);
    }

    public function mapAndConditions(array $conditions, ?string $alias = null): static
    {
        $alias = $alias ?: $this->getMainAlias();

        foreach ($conditions as $name => $value) {
            $parts = explode('.', $name, 2);
            if (!isset($parts[1])) {
                $name = $alias . '.' . $name;
            }
            $this->bindAndCondition($name, $value);
        }

        return $this;
    }

    public function selectCount(): static
    {
        return $this->select('COUNT(' . $this->getMainAlias() . ')')
            ->setMaxResults(1);
    }

    public function setFrameResults(?int $start = null, ?int $limit = null): static
    {
        $start = max(0, (int) $start);
        $limit = max(0, (int) $limit);

        if (0 < $start) {
            $this->setFirstResult($start);
        }

        if (0 < $limit) {
            $this->setMaxResults($limit);
        }

        return $this;
    }

    public function bindAndCondition(string $name, $value, string $type = '='): static
    {
        $placeholder = str_replace('.', '_', $name);

        return $this->andWhere($name . ' ' . $type . ' :' . $placeholder)
            ->setParameter($placeholder, $value);
    }

    public function bindOrCondition(string $name, $value, string $type = '='): static
    {
        $placeholder = str_replace('.', '_', $name);

        return $this->orWhere($name . ' ' . $type . ' :' . $placeholder)
            ->setParameter($placeholder, $value);
    }
}
