<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Bundle\DoctrineBridgeBundle\Enricher\GetList\Filter\Assembler;

use ReflectionException;
use ReflectionObject;
use ReflectionProperty;
use RuntimeException;
use XCart\Bundle\CommonBundle\Assembler\PropertyNameAssembler;
use XCart\Bundle\DoctrineBridgeBundle\DTO\Filter\FilterInterface;
use XCart\Bundle\DoctrineBridgeBundle\DTO\Filter\Value\FilterValueInterface;
use XCart\Bundle\DoctrineBridgeBundle\Enricher\GetList\Assembler\FieldNameAssemblerInterface;
use XCart\Bundle\DoctrineBridgeBundle\Enricher\GetList\Filter\DTO\FilterListElement;
use XCart\Bundle\DoctrineBridgeBundle\Enricher\GetList\Filter\Resolver\FilterApplierResolverInterface;

class FilterListAssembler implements FilterListAssemblerInterface
{
    public function __construct(
        private FilterApplierResolverInterface $applierResolver,
        private FieldNameAssemblerInterface $fieldNameAssembler,
    ) {
    }

    public function assemble(FilterInterface $filter): array
    {
        $result = [];

        foreach ($this->getValuesList($filter) as $property => $value) {
            $applier = $this->applierResolver->resolve($property, $value);
            if (!$applier) {
                continue;
            }
            $filterListElement = (new FilterListElement())
                ->setApplier($applier)
                ->setProperty($this->fieldNameAssembler->assemble($property))
                ->setValue($value);

            $result[] = $filterListElement;
        }

        return $result;
    }

    /**
     * @return array<string,FilterValueInterface>
     */
    private function getValuesList(FilterInterface $filter): array
    {
        $reflection = new ReflectionObject($filter);

        $result = [];

        foreach ($reflection->getMethods() as $method) {
            if (!str_starts_with($method->getName(), 'get')) {
                continue;
            }

            try {
                /** @var ?FilterValueInterface $value */
                $value = $method->invoke($filter);
            } catch (ReflectionException) {
                continue;
            }
            if (!($value instanceof FilterValueInterface)) {
                continue;
            }

            $property = $this->detectPropertyName(
                $reflection->getProperties(ReflectionProperty::IS_PRIVATE),
                substr($method->getName(), 3),
            );

            $result[$property] = $value;
        }

        return $result;
    }

    /**
     * @param list<ReflectionProperty> $properties
     */
    private function detectPropertyName(array $properties, string $methodName): string
    {
        $propertyName = lcfirst($methodName);

        foreach ($properties as $property) {
            if ($property->getName() === $propertyName) {
                return $propertyName;
            }
        }

        $propertyName = PropertyNameAssembler::assembleSnakePropertyName($methodName);

        foreach ($properties as $property) {
            if ($property->getName() === $propertyName) {
                return $propertyName;
            }
        }

        throw new RuntimeException(sprintf('Cannot detect filter\'s property "%s"', $propertyName));
    }
}
