<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DoctrineBridgeBundle\Decorator\Transactional;

use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Throwable;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\Assembler\Response\ExceptionalFailedResponseAssemblerInterface;
use XCart\Bundle\LogicBundle\DTO\Response\FailedResponseInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;

class TransactionalDecorator implements ActionInterface
{
    use LoggerAwareTrait;

    public function __construct(
        private ActionInterface $inner,
        private EntityManagerInterface $writeEntityManager,
        private ExceptionalFailedResponseAssemblerInterface $failedResponseAssembler,
        LoggerInterface $logger,
    ) {
        $this->setLogger($logger);
    }

    public function run(RequestInterface $request): ResponseInterface
    {
        $this->writeEntityManager->beginTransaction();

        $response = $this->inner->run($request);

        if ($response instanceof FailedResponseInterface) {
            $this->writeEntityManager->rollback();

            return $response;
        }

        try {
            $this->writeEntityManager->commit();
        } catch (Throwable $e) {
            $this->logger->error(
                'Request {{requestType}} was successful, but transactional failed with {{message}} error',
                [
                    'requestType' => get_class($request),
                    'message'     => (string)$e,
                    'exception'   => $e,
                ],
            );

            return $this->failedResponseAssembler->assemble(
                $e,
                sprintf('Request "%s" was successful, but transactional failed', get_class($request)),
            );
        }

        return $response;
    }
}
