<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Bundle\DoctrineBridgeBundle\Action;

use InvalidArgumentException;
use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use Throwable;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\UpdateOne\Arguments\ArgumentsAssemblerInterface;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetOne\Response\ResponseAssemblerInterface;
use XCart\Bundle\DoctrineBridgeBundle\Enricher\UpdateOne\EntityEnricherInterface;
use XCart\Bundle\DoctrineBridgeBundle\Repository\WriteRepositoryInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;

class UpdateOneAction implements ActionInterface
{
    public function __construct(
        private WriteRepositoryInterface $writeRepository,
        private ArgumentsAssemblerInterface $argumentsAssembler,
        private EntityEnricherInterface $entityEnricher,
        private ResponseAssemblerInterface $responseAssembler,
    ) {
    }

    public function run(RequestInterface $request): ResponseInterface
    {
        /** @noinspection PhpParamsInspection */
        $arguments = $this->argumentsAssembler->assemble($request);

        try {
            $entity = $this->writeRepository->findOneBy($arguments);
        } catch (Throwable $e) {
            if ($e instanceof InvalidArgumentException) {
                return (new DefaultFailedResponse())
                    ->setType('Bad request')
                    ->setTitle(sprintf('Cannot assemble entity: %s', $e->getMessage()))
                    ->setStatus(HTTPResponse::HTTP_BAD_REQUEST);
            }

            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle(sprintf('Cannot read entity from DB: %s', $e->getMessage()))
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        if (!$entity) {
            return (new DefaultFailedResponse())
                ->setType('Not found')
                ->setTitle('Cannot find entity')
                ->setStatus(HTTPResponse::HTTP_NOT_FOUND);
        }

        /** @noinspection PhpParamsInspection */
        $entity = $this->entityEnricher->enrich($entity, $request);

        try {
            $this->writeRepository->updateEntity($entity);
        } catch (Throwable $e) {
            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle(sprintf('Cannot update entity: %s', $e))
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        /** @noinspection PhpIncompatibleReturnTypeInspection */
        return $this->responseAssembler->assemble($entity);
    }
}
