<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\DoctrineBridgeBundle\Action;

use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use Throwable;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetList\Criteria\FindListCriteriaAssemblerInterface;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetList\Response\ResponseAssemblerInterface;
use XCart\Bundle\DoctrineBridgeBundle\DataSource\DTO\CountListCriteria;
use XCart\Bundle\DoctrineBridgeBundle\DataSource\ReadDataSourceInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\PaginatorRequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\PageOwnerInterface;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;

class GetListAction implements ActionInterface
{
    public function __construct(
        private ReadDataSourceInterface $readDataSource,
        private FindListCriteriaAssemblerInterface $criteriaAssembler,
        private ResponseAssemblerInterface $responseAssembler,
    ) {
    }

    public function run(RequestInterface $request): ResponseInterface
    {
        try {
            $criteria = $this->criteriaAssembler->assemble($request);
        } catch (Throwable $e) {
            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle(sprintf('Cannot assemble search criteria: %s', $e))
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        try {
            $entities = $this->readDataSource->findList($criteria);
        } catch (Throwable $e) {
            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle(sprintf('Cannot read entities from DB: %s', $e))
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        $response = $this->responseAssembler->assemble($entities->getValues());
        if (
            $request instanceof PaginatorRequestInterface
            && $response instanceof PageOwnerInterface
            && $request->getItemsPerPage() > 0
        ) {
            $countCriteria = (new CountListCriteria())
                ->setFilter($criteria->getFilter())
                ->setOrderRule($criteria->getOrderRule());
            try {
                $response->setTotalItems($this->readDataSource->countList($countCriteria));
            } catch (Throwable $e) {
                return (new DefaultFailedResponse())
                    ->setType('Internal error')
                    ->setTitle(sprintf('Cannot read entities count from DB: %s', $e))
                    ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
            }

            $response
                ->setPageIndex($request->getPageIndex())
                ->setItemsPerPage($request->getItemsPerPage());
        }

        return $response;
    }
}
