<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\CommonBundle\Serializer\Handler;

use InvalidArgumentException;
use JMS\Serializer\Context;
use JMS\Serializer\GraphNavigatorInterface;
use JMS\Serializer\Handler\SubscribingHandlerInterface;
use JMS\Serializer\Visitor\DeserializationVisitorInterface;
use JMS\Serializer\Visitor\SerializationVisitorInterface;
use MyCLabs\Enum\Enum;

final class EnumHandler implements SubscribingHandlerInterface
{
    public const TYPE_ENUM = 'enum';

    public static function getSubscribingMethods(): array
    {
        $methods = [];
        foreach (['json', 'xml', 'yml'] as $format) {
            $methods[] = [
                'direction' => GraphNavigatorInterface::DIRECTION_SERIALIZATION,
                'type'      => self::TYPE_ENUM,
                'format'    => $format,
                'method'    => 'serializeEnum',
            ];
            $methods[] = [
                'direction' => GraphNavigatorInterface::DIRECTION_DESERIALIZATION,
                'type'      => self::TYPE_ENUM,
                'format'    => $format,
                'method'    => 'deserializeEnum',
            ];
        }

        return $methods;
    }

    /**
     * @throws InvalidArgumentException
     */
    public function serializeEnum(SerializationVisitorInterface $visitor, Enum $enum, array $typeMetadata, Context $context)
    {
        $value = $enum->getValue();

        if (is_int($value)) {
            return $visitor->visitInteger($value, $typeMetadata);
        }

        if (is_string($value)) {
            return $visitor->visitString($value, $typeMetadata);
        }

        if (is_bool($value)) {
            return $visitor->visitBoolean($value, $typeMetadata);
        }

        if (is_float($value)) {
            return $visitor->visitDouble($value, $typeMetadata);
        }

        throw new InvalidArgumentException(
            sprintf('Enumerate serialization not support "%s" type', gettype($value))
        );
    }

    public function deserializeEnum(DeserializationVisitorInterface $visitor, $data, array $type, Context $context): Enum
    {
        $enumClass = $this->getEnumClass($type);

        return new $enumClass($data);
    }

    /**
     * @return class-string<Enum>
     */
    private function getEnumClass(array $type): string
    {
        return $type['params'][0]['name'];
    }
}
