<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\CommonBundle\Resolver\Mapper;

use ArrayIterator;
use IteratorAggregate;
use Traversable;
use XCart\Bundle\CommonBundle\Resolver\Exception\KeyTypeWrongException;
use XCart\Bundle\CommonBundle\Resolver\Exception\ValueTypeWrongException;

class HashMapper implements IteratorAggregate, HashMapperInterface
{
    private array $list = [];

    private mixed $default;

    private string $keyType;

    private string $valueType;

    /**
     * @throws KeyTypeWrongException
     * @throws ValueTypeWrongException
     */
    public function __construct(array $list, mixed $default = null, string $keyType = 'string', string $valueType = null)
    {
        foreach ($list as $key => $value) {
            $this->add($key, $value);
        }
        $this->default = $default;
        $this->keyType = $keyType;
        $this->valueType = $valueType;
    }

    /**
     * @throws KeyTypeWrongException
     * @throws ValueTypeWrongException
     */
    private function add(mixed $key, mixed $value): void
    {
        if ($this->keyType && $this->getType($key) !== $this->keyType) {
            throw KeyTypeWrongException::build($this->getType($key), $this->keyType);
        }

        if ($this->valueType && $this->getType($value) !== $this->valueType) {
            throw ValueTypeWrongException::build($this->getType($value), $this->valueType);
        }

        $this->list[$key] = $value;
    }

    public function map(mixed $key): mixed
    {
        return $this->list[$key] ?? $this->default;
    }

    public function getIterator(): Traversable
    {
        return new ArrayIterator($this->list);
    }

    private function getType(mixed $value): string
    {
        $type = gettype($value);

        if ($type === 'object') {
            return get_class($value);
        }

        return $type;
    }
}
