<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\CommonBundle\Enricher\Transparent;

use Generator;
use ReflectionException;
use ReflectionMethod;
use ReflectionObject;
use XCart\Bundle\CommonBundle\Assembler\MethodNameAssembler;

class WalkingTransparentEnricher implements TransparentEnricherInterface
{
    public function enrich(object $initial, object $input): object
    {
        $initialReflection = new ReflectionObject($initial);
        $inputReflection = new ReflectionObject($input);

        foreach ($this->getGetters($inputReflection) as $getter => $method) {
            $setter = $this->getSetter($initialReflection, $getter);
            if (!$setter) {
                continue;
            }

            try {
                $value = $method->invoke($input);
            } catch (ReflectionException) {
                continue;
            }

            switch (gettype($value)) {
                case 'array':
                    $initialGetter = $this->getGetter($initialReflection, $getter);
                    if ($initialGetter) {
                        try {
                            /** @noinspection SlowArrayOperationsInLoopInspection */
                            $value = array_merge($value, $initialGetter->invoke($initial));
                        } catch (ReflectionException) {
                            continue 2;
                        }
                    }
                    break;

                case 'object':
                    if (!(new ReflectionObject($value))->isInternal()) {
                        $initialGetter = $this->getGetter($initialReflection, $getter);
                        if ($initialGetter) {
                            try {
                                $value = $this->enrich($initialGetter->invoke($initial), $value);
                            } catch (ReflectionException) {
                                continue 2;
                            }
                        }
                    }
                    break;
            }

            try {
                $setter->invoke($initial, $value);
            } catch (ReflectionException) {
            }
        }

        return $initial;
    }

    /**
     * @param ReflectionObject $reflection
     *
     * @return Generator<string,ReflectionMethod>
     */
    private function getGetters(ReflectionObject $reflection): Generator
    {
        foreach ($reflection->getMethods() as $method) {
            if ($method->isPublic() && !$method->isAbstract() && MethodNameAssembler::isGetter($method->getName())) {
                yield $method->getName() => $method;
            }
        }
    }

    private function getGetter(ReflectionObject $reflection, $getterMethodName): ?ReflectionMethod
    {
        try {
            return $reflection->getMethod($getterMethodName);
        } catch (ReflectionException) {
            return null;
        }
    }

    private function getSetter(ReflectionObject $reflection, $getterMethodName): ?ReflectionMethod
    {
        $setterName = MethodNameAssembler::assembleSetterName(
            MethodNameAssembler::getPropertyNameFromGetterName($getterMethodName)
        );

        foreach ($reflection->getMethods() as $method) {
            if (
                $method->isPublic()
                && !$method->isAbstract()
                && $setterName === $method->getName()
            ) {
                return $method;
            }
        }

        return null;
    }
}
