<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\Filter;

use XCart\Bundle\LogicBundle\DTO\Request\Filter\Value\FilterValueInterface;
use XCart\Bundle\LogicBundle\DTO\Request\Filter\Value\FloatValue;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\FilterStrategyInterface;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\NumberFilterStrategyType;

class FloatFilter extends AbstractTypedFilter
{
    protected function buildDeclaration(string $name, string $id): array
    {
        $result = parent::buildDeclaration($name, $id);

        $result['schema']['format'] = 'float';

        return $result;
    }

    protected function buildStrategyDeclaration(string $name, string $id, array $strategies): array
    {
        $result = parent::buildStrategyDeclaration($name, $id, $strategies);

        $result['schema']['default'] = NumberFilterStrategyType::EXACT;

        return $result;
    }

    /**
     * @param float                    $value
     * @param NumberFilterStrategyType $strategy
     * @param bool                     $exclude
     *
     * @return FloatValue
     */
    protected function cast(mixed $value, FilterStrategyInterface $strategy, bool $exclude): FilterValueInterface
    {
        return (new FloatValue())
            ->setValue((float) $value)
            ->setStrategy($strategy)
            ->setExclude($exclude);
    }

    protected function getType(): string
    {
        return 'float';
    }

    protected function getSwaggerType(): string
    {
        return 'number';
    }

    protected function assembleStrategy(?string $strategyCode): FilterStrategyInterface
    {
        if (!empty($strategyCode)) {
            return NumberFilterStrategyType::from($strategyCode);
        }

        if ($this->strategies) {
            reset($this->strategies);
            $firstStrategy = current($this->strategies);

            return NumberFilterStrategyType::from($firstStrategy);
        }

        return NumberFilterStrategyType::EXACT();
    }
}
