<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\Filter;

use ApiPlatform\Core\Exception\InvalidArgumentException;
use XCart\Bundle\LogicBundle\DTO\Request\Filter\Value\FilterValueInterface;
use XCart\Bundle\LogicBundle\DTO\Request\Filter\Value\BooleanValue;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\BooleanFilterStrategyType;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\FilterStrategyInterface;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\NumberFilterStrategyType;

class BooleanFilter extends AbstractTypedFilter
{
    protected function buildDeclaration(string $name, string $id): array
    {
        $result = parent::buildDeclaration($name, $id);

        $result['schema']['example'] = '1';

        return $result;
    }

    protected function buildStrategyDeclaration(string $name, string $id, array $strategies): array
    {
        $result = parent::buildStrategyDeclaration($name, $id, $strategies);

        $result['schema']['default'] = NumberFilterStrategyType::EXACT;

        return $result;
    }

    /**
     * @param int                       $value
     * @param BooleanFilterStrategyType $strategy
     * @param bool                      $exclude
     *
     * @return BooleanValue
     */
    protected function cast(mixed $value, FilterStrategyInterface $strategy, bool $exclude): FilterValueInterface
    {
        if (in_array($value, ['0', 'false', 'FALSE'], true)) {
            $convertedValue = false;
        } elseif (in_array($value, ['1', 'true', 'TRUE'], true)) {
            $convertedValue = true;
        } else {
            throw new InvalidArgumentException(
                "\"{$value}\" has a wrong format. Value should be a boolean ('0', '1', 'false', 'true')"
            );
        }

        return (new BooleanValue())
            ->setValue($convertedValue)
            ->setStrategy($strategy)
            ->setExclude($exclude);
    }

    protected function getType(): string
    {
        return 'bool';
    }

    protected function getSwaggerType(): string
    {
        return 'boolean';
    }

    protected function assembleStrategy(?string $strategyCode): FilterStrategyInterface
    {
        return BooleanFilterStrategyType::EXACT();
    }
}
