<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\Filter;

use XCart\Bundle\LogicBundle\Enum\FilterStrategy\FilterStrategyInterface;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\RangeFilterStrategyType;

abstract class AbstractTypedRangeFilter extends AbstractFilter
{
    protected const KEY_TEMPLATE          = 'filter.%s.range';
    protected const KEY_STRATEGY_TEMPLATE = 'filter.%s.range.strategy';
    protected const KEY_EXCLUDE_TEMPLATE  = 'filter.%s.range.exclude';

    protected function buildDeclaration(string $name, string $id): array
    {
        return [
            'property'    => $id,
            'type'        => $this->getType(),
            'required'    => $this->required,
            'description' => sprintf('Range filter for "%s.range" field', $name),
            'schema'      => [
                'name'       => $id,
                'type'       => 'object',
                'properties' => [
                    'begin' => [
                        'type'     => $this->getSwaggerType(),
                        'nullable' => !$this->required,
                    ],
                    'end'   => [
                        'type'     => $this->getSwaggerType(),
                        'nullable' => !$this->required,
                    ],
                ],
            ],
        ];
    }

    protected function buildStrategyDeclaration(string $name, string $id, array $strategies): array
    {
        return [
            'property'    => $id,
            'type'        => 'string',
            'required'    => false,
            'description' => sprintf('Filter strategy for "%s.range" field', $name),
            'schema'      => [
                'name'    => $id,
                'type'    => 'string',
                'enum'    => $strategies,
                'default' => RangeFilterStrategyType::INCLUDE_BOTH(),
            ],
        ];
    }

    protected function buildExcludeDeclaration(string $name, string $id): array
    {
        return [
            'property'    => $id,
            'type'        => 'boolean',
            'required'    => false,
            'description' => sprintf('Filter exclude flag for "%s.range" field', $name),
            'schema'      => [
                'name'    => $id,
                'type'    => 'boolean',
                'default' => false,
            ],
        ];
    }

    protected function assembleStrategy(?string $strategyCode): FilterStrategyInterface
    {
        if (!empty($strategyCode)) {
            return RangeFilterStrategyType::from($strategyCode);
        }

        if ($this->strategies) {
            reset($this->strategies);
            $firstStrategy = current($this->strategies);

            return RangeFilterStrategyType::from($firstStrategy);
        }

        return RangeFilterStrategyType::INCLUDE_BOTH();
    }
}
