<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\Filter;

use XCart\Bundle\LogicBundle\Enum\FilterStrategy\FilterStrategyInterface;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\ListFilterStrategyType;

abstract class AbstractTypedListFilter extends AbstractFilter
{
    protected const KEY_TEMPLATE          = 'filter.%s.list';
    protected const KEY_STRATEGY_TEMPLATE = 'filter.%s.list.strategy';
    protected const KEY_EXCLUDE_TEMPLATE  = 'filter.%s.list.exclude';

    protected function buildDeclaration(string $name, string $id): array
    {
        return [
            'property'      => $id,
            'type'          => $this->getType(),
            'required'      => $this->required,
            'is_collection' => true,
            'description'   => sprintf('Filter for "%s.list" field', $name),
            'schema'        => [
                'name'      => $id,
                'type'      => 'array',
                'minLength' => $this->required ? 1 : 0,
                'items'     => [
                    'type' => $this->getSwaggerType(),
                ],
            ],
        ];
    }

    protected function buildStrategyDeclaration(string $name, string $id, array $strategies): array
    {
        return [
            'property'    => $id,
            'type'        => 'string',
            'required'    => false,
            'description' => sprintf('Filter strategy for "%s.list" field', $name),
            'schema'      => [
                'name'    => $id,
                'type'    => 'string',
                'enum'    => $strategies,
                'default' => ListFilterStrategyType::IN,
            ],
        ];
    }

    protected function buildExcludeDeclaration(string $name, string $id): array
    {
        return [
            'property'    => $id,
            'type'        => 'boolean',
            'required'    => false,
            'description' => sprintf('Filter exclude flag for "%s.list" field', $name),
            'schema'      => [
                'name'    => $id,
                'type'    => 'boolean',
                'default' => false,
            ],
        ];
    }

    protected function assembleStrategy(?string $strategyCode): FilterStrategyInterface
    {
        if (!empty($strategyCode)) {
            return ListFilterStrategyType::from($strategyCode);
        }

        if ($this->strategies) {
            reset($this->strategies);
            $firstStrategy = current($this->strategies);

            return ListFilterStrategyType::from($firstStrategy);
        }

        return ListFilterStrategyType::IN();
    }
}
