<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\Filter;

use ApiPlatform\Doctrine\Common\Filter\SearchFilterInterface;
use XCart\Bundle\LogicBundle\DTO\Request\Filter\FilterInterface as LogicFilterInterface;
use XCart\Bundle\CommonBundle\Assembler\MethodNameAssembler;
use XCart\Bundle\LogicBundle\DTO\Request\Filter\Value\FilterValueInterface;
use XCart\Bundle\LogicBundle\Enum\FilterStrategy\FilterStrategyInterface;

abstract class AbstractFilter implements FilterInterface, SearchFilterInterface, OperationSpecificFilterInterface
{
    protected const KEY_TEMPLATE = 'filter.%s';
    protected const KEY_STRATEGY_TEMPLATE = 'filter.%s.strategy';
    protected const KEY_EXCLUDE_TEMPLATE = 'filter.%s.exclude';

    protected array $properties = [];

    protected bool $defaultExclude = false;

    /**
     * @param string[] $operations
     */
    public function __construct(
        array $properties = [],
        protected array $strategies = [],
        protected bool $allowExclude = false,
        protected bool $required = false,
        protected array $operations = [],
    ) {
        $this->properties = array_keys($properties);
    }

    public function enrich(
        LogicFilterInterface $filter,
        string $resourceClass,
        string $operationName,
        array $context,
    ): void {
        foreach ($this->properties as $name) {
            $key = sprintf(static::KEY_TEMPLATE, $name);
            $strategyKey = sprintf(static::KEY_STRATEGY_TEMPLATE, $name);
            $excludeKey = sprintf(static::KEY_EXCLUDE_TEMPLATE, $name);
            if (isset($context['filters'][$key])) {
                $strategy = $this->assembleStrategy($context['filters'][$strategyKey] ?? null);
                $exclude = $context['filters'][$excludeKey] ?? $this->defaultExclude;

                $setterName = MethodNameAssembler::assembleSetterName($name);
                $filter->$setterName(
                    $this->cast(
                        $context['filters'][$key],
                        $strategy,
                        $exclude,
                    ),
                );
            }
        }
    }

    public function getDescription(string $resourceClass): array
    {
        $result = [];

        foreach ($this->properties as $name) {
            $id = sprintf(static::KEY_TEMPLATE, $name);
            $result[$id] = $this->buildDeclaration($name, $id);
            if (count($this->strategies) > 1) {
                $id = sprintf(static::KEY_STRATEGY_TEMPLATE, $name);
                $result[$id] = $this->buildStrategyDeclaration($name, $id, $this->strategies);
            }
            if ($this->allowExclude) {
                $id = sprintf(static::KEY_EXCLUDE_TEMPLATE, $name);
                $result[$id] = $this->buildExcludeDeclaration($name, $id);
            }
        }

        return $result;
    }

    public function getOperations(): array
    {
        return $this->operations;
    }

    protected function getSwaggerType(): string
    {
        return $this->getType();
    }

    protected function getOpenAPIType(): string
    {
        return $this->getSwaggerType();
    }

    abstract protected function cast(mixed $value, FilterStrategyInterface $strategy, bool $exclude): FilterValueInterface;

    abstract protected function getType(): string;

    abstract protected function assembleStrategy(?string $strategyCode): FilterStrategyInterface;

    abstract protected function buildDeclaration(string $name, string $id): array;

    abstract protected function buildStrategyDeclaration(string $name, string $id, array $strategies): array;

    abstract protected function buildExcludeDeclaration(string $name, string $id): array;
}
